*! version 1.0.0  25aug2014  dcs

* leans on code from official Stata's -svar-

program define svarih, nclass // do not sortpreserve here as this interferes with datasig in -svarih examples-

    version 11.2

    if replay() {  // replay must be invoked by "svarih, ..." ; for "svarih METHOD, ..." an error is issued
                   //        since the parsing becomes a little too complicated
        if `"`e(cmd)'"'=="svarih" {
            local subcmd = strlower("`e(method)'")
        }
        else {
            exit 301
        }
        // local rest : copy local 0
        
        syntax , [cmat *]
        local rest `", `options'"'   // must take cmat out of `0' in all cases: may be "cmat" or "nocmat" ; the latter would cause an error in regular replay

        if "`cmat'"!="" {
            
            syntax , [Full noTable noCNSReport *]
            _get_diopts diopts jnk , `options'
            
            if `"`full'`table'`cnsreport'`diopts'"'!="" {
                disp as error `"svarih replay: option 'cmat' can only be accompanied with options from {help svarih cmat}"'
                exit 198
            }

            svarih_cmat `rest'
            exit
        }
        
    }
    else {
        local subcmd : word 1 of `0'
        local 0 : subinstr local 0 `"`subcmd'"' ""
        local subcmd = lower(`"`subcmd'"')
        
        local endcomma = substr(`"`subcmd'"',length(`"`subcmd'"'),1)==","
        if replay() | `endcomma' {  // replay() checks local `0'
            disp as error `"Invoke estimation results replay without specifying a subcommand."'
            exit 198
        }
        local rest `"`0'"'
        
        local mlv : disp "version " string(_caller()) ":"  // important to run -ml- under version control b/c of changes from Stata 11.2 to 12
        local mlvopt mlv(`mlv')
    }

    local lsubcmd = length(`"`subcmd'"')


    if `"`subcmd'"' == substr("bacchiocchi",1,max(3,`lsubcmd')) {
        qui _ts , sort onepanel
        svarih_bac `rest' `mlvopt'
    }
    else if `"`subcmd'"' == substr("bfanelli",1,max(3,`lsubcmd')) {
        qui _ts , sort onepanel
        svarih_bfa `rest' `mlvopt'
    }
    else if `"`subcmd'"' == substr("llutkepohl",1,max(3,`lsubcmd')) {
        qui _ts , sort onepanel
        svarih_llu `rest' `mlvopt'
    }
    else if `"`subcmd'"' == substr("examples",1,max(3,`lsubcmd')) {
        svarih_examples `rest'
    }
    else {
        disp as error `"Subcmd '`subcmd'' not recognized."'
        exit 198
    }

end

*** ----------------- SUBROUTINES COMMON TO ALL SVARIH SUBCOMMANDS ------------------

program define _svarih_mkmatcns, rclass
// official Stata's -svar- subroutine _svar_mkmatcns

        syntax , mat(string)   ///  `mat' will contain the option input (name of existing matrix or string matrix definition)
                 name(string)  ///  name of equation of ML parms ; sometimes shown in error messages
                 neqs(integer) ///
                 tname(string) ///  `tname' will contain the matrix that is newly defined by the option input
                 type(string)  ///  only relevant for error messages
                 [ eq ]        //   for options aeq, beq, leq, etc.
        
        _svarih_PARSEab, mat(`mat')     ///
                         name(`name')   ///
                         neqs(`neqs')   ///
                         tname(`tname') //
        
        capture assert rowsof(`tname') == `neqs'
        if _rc > 0 {
            di as err "`type' matrix does not have the correct row dimension"
            exit 198
        }       
        capture assert colsof(`tname') == `neqs'
        if _rc > 0 {
            di as err "`type' matrix does not have the correct column dimension"
            exit 198
        }       
        if "`eq'" == "" {
            _svarih_cnsmac, mat(`tname') name(`name') neqs(`neqs')
        }
        else {
            _svarih_eqmac , mat(`tname') name(`name') neqs(`neqs') 
        }
        local cnsmac "`r(cnsmac)'"
        gettoken tok cnsmac: cnsmac ,parse(":") 
        if "`cnsmac'" == "" {
            di as err "no constraints implied by matrix `a'"
            exit 498
        }   
        while "`cnsmac'" != "" {
            gettoken tok cnsmac:cnsmac, parse(":")
            if "`tok'" != ":" {
                _svarih_newcns `tok'
                local svar_cnslist `svar_cnslist' `r(new)'

            }
        }
        ret local svar_cnslist `svar_cnslist'
end         

program define _svarih_PARSEab
// modified official Stata's -svar- subroutine _PARSEab
    syntax , mat(string) name(string) neqs(integer) tname(string)
    
    * local toks : word count `mat'
    capture confirm name `mat'
    if !_rc {
        capture confirm matrix `mat'
        if _rc > 0 {
            di as err "`name'(`mat') does not define a matrix"
            exit 198
        }   
        mat `tname' = `mat'
    }
    else {
        * capture matrix input `tname' = `mat'  // "input" prevents expressions like I() or J()
        capture matrix `tname' = `mat'
        if _rc > 0 {
            di as err "`name'(`mat') does not define a matrix"
            exit 198
        }   
    }
end 

program define _svarih_cnsmac, rclass
// copypaste of official Stata's _svar_cnsmac.ado, version 1.0.1  21aug2002

    version 11.2

    syntax , mat(string) name(string) neqs(integer) 

    tempname el
    local k 0
    forvalues i = 1/`neqs' {
        forvalues j = 1/`neqs' {
            scalar `el' = `mat'[`i',`j']
            if `el' != int(`el') {
                di as err "non-integer element in matrix defining constraints on `name'"
                exit 198    
            }   

            if `el' < . {
                if `el' == 0 {
                    local cnsmac "`cnsmac':[`name'_`i'_`j']_cons = 0"               
                }
                else {
                    local already 0             
                    forvalues m = 1/`k' {
                        if ``m'' == `el' {
                            local already 1
                            local `m'c ``m'c' `name'_`i'_`j'
                        }
                    }
                    if `already' == 0 {
                        local ++k
                        local `k' = `el'
                        local `k'c `name'_`i'_`j'
                    }   
                }
            }
        }
    }

    forvalue m = 1/`k' {
        local cns : word count ``m'c'
        if `cns' < 1 {
            di as err "_svarih_cnsmac is broken"
            exit 498
        }   
        if `cns' == 1 {
            di as err "`mc'c' is not constrained to be equal to another element"
            exit 498
        }   
        local left ``m'c'
        gettoken cur left:left
        local first `cur'
        forvalues n = 2/`cns' {
            gettoken cur left:left
            local contr "[`first']_cons = [`cur']_cons"
            local cnsmac "`cnsmac':`contr'"
        }   
    }
    return local cnsmac "`cnsmac'"
end

program define _svarih_eqmac, rclass
// copypaste of official Stata's _svar_eqmac.ado, version 1.0.0  20aug2002

    version 11.2

    syntax , mat(string) name(string) neqs(integer) 

    forvalues i = 1/`neqs' {
        forvalues j = 1/`neqs' {
            local el = `mat'[`i',`j']

            if `el' < . {
                local cnsmac "`cnsmac':[`name'_`i'_`j']_cons = `el'"                
            }

        }
    }
    return local cnsmac "`cnsmac'"
end

program define _svarih_newcns, rclass
// copypaste of official Stata's _svar_newcns.ado, version 1.1.0  22aug2002
    
    version 11.2
    
    syntax anything(name=cns id="constraint" equalok)

    constraint free
    local free = r(free)
    constraint define `free' `cns'
    return local new `free'
end

program define _svarih_Dheadernew
// official Stata's -svar- subroutine _Dheadernew, modified to suit -svarih-

    syntax [, noCNSReport]
    local neqs = e(k_dv)

    di
    di as txt "Structural vector autoregression, IH method `e(method)'"
    di

    if "`cnsreport'" != "nocnsreport" {
        matrix dispCns, r
        if r(k) > 0 {
            matrix dispCns
            di
        }       
    }

    // display sample span
    if "`e(method)'"=="BFanelli" {
        local tmin `e(tmin_var1)'
        local tmax `e(tmax_var1)'
        local N_gaps 0
        forvalues rgm=1/`e(numregimes)' {
            local N_gaps = `N_gaps' + `e(N_gaps_var`rgm')'
            if `e(tmin_var`rgm')'<`tmin' local tmin `e(tmin_var`rgm')'
            if `e(tmax_var`rgm')'>`tmax' local tmax `e(tmax_var`rgm')'
        }
        if `N_gaps' > 0 {
            if `N_gaps' == 1 {
                local gapstr ", but with at least one gap"
            }
            else {
                local gapstr ", but with at least `N_gaps' gaps"
            }
        }
    }
    else {
        local tmin `e(tmin)'
        local tmax `e(tmax)'
        local N_gaps = e(N_gaps)
        if `N_gaps' > 0 {
            if `N_gaps' == 1 {
                local gapstr ", but with a gap"
            }
            else {
                local gapstr ", but with `N_gaps' gaps"
            }
        }
    }
    
    local tmin `=trim("`: disp `e(tsfmt)' `tmin''")'
    local tmax `=trim("`: disp `e(tsfmt)' `tmax''")'

    disp as text "Sample span  : " as res "`tmin'" as text " - " as res "`tmax'" "`gapstr'"
    
    if "`e(method)'"=="LLutkepohl" {
        disp ""
        disp as text "Ident. check"
        disp as text "min. Wald      = " as res %9.4g e(Wald_min) as text "{col 52}No. of obs{col 68}= "      as res %9.0f e(N)  
        disp as text "Prob > chi2(1) = " as res %9.3f e(Wald_p)   as text "{col 52}Log likelihood{col 68}= "  as res %9.8g e(ll)  
    }
    else {
        disp as text "Ident. check : " as res "`e(idencheck)'"
        disp
        disp as text "No. of obs   = " as res %-9.0f e(N) "{col 52}Log likelihood{col 68}= "  as res %9.8g e(ll)  
    }
end

program define _svarih_ETable2, eclass
// official Stata's -svar- subroutine Etable2, modified to suit -svarih-

    version 11.2
    
    syntax , level(cilevel) [ full  dfr(numlist max = 1 >0) *]
    tempname b v bt vt esample rowt eres
        
    _get_diopts options, `options'
    mat `b' = e(b)
    mat `v' = e(V)
    if "`e(Cns)'" == "matrix" {
        tempname C
        matrix `C' = e(Cns)
    }

    local k_eq  = e(k_eq)
    local k_aux = e(k_aux)
    local sep   = e(k_dv)^2
        
    nobreak {
        capture _est hold `eres', varname(`esample') restore
        ereturn post `b' `v' `C'
        if "`dfr'" != "" {
            ereturn scalar df_r = `dfr'
        }   

        ereturn scalar k_eq  = `k_eq'
        ereturn scalar k_aux = `k_aux'

        ereturn local cmd "svarih"

        _coef_table , level(`level') separator(`sep') `options' nocnsreport
        _est unhold `eres'
    }

end 

*** ----------------- END COMMON SUBROUTINES ------------------




program define svarih_bac, nclass

    version 11.2

    if replay() {
        if "`e(cmd)'" != "svarih" | "`e(method)'"!="Bacchiocchi" error 301

        if _by() {
            error 190 
        }

        syntax [ , Level(cilevel) Full noTable noCNSReport *]
        
        _get_diopts options, `options'
        _svarih_Dheadernew, `cnsreport'
        
        if "`table'"!="notable" {
            if "`e(small)'" != "" {
                local dfr = e(df_r)
                _svarih_ETable2, level(`level') dfr(`dfr') `options'
            }
            else {
                _svarih_ETable2, level(`level') `options'
            }   
        }
        exit
    } 
    else {
        svarih_bac_est `0'
    }
end

*** ------------------------ svarih_bac_est ---------------------------------------------------

program define svarih_bac_est, eclass sortpreserve

    version 11.2

    syntax varlist [if] [in]                    ///
                                                    /// --- svarih_bac-related ---
           , RGMVar(varname)                         /// variable that identifies the regime periods
             RGMMat(string)                          /// numregimes x numendog matrix that defines the regimes
           [ ACONstraints(numlist >0 int)       /// 
             AEq(string)                        /// 
             ACns(string)                       /// 
             BCONstraints(numlist >0 int)       /// 
             BEq(string)                         /// 
             BCns(string)                        /// 
             ECONstraints(numlist >0 int)        /// 
             EEq(string)                         /// 
             ECns(string)                        /// 
             noIDENcheck                         /// 
             IDENDetail                          ///
                                                    /// 
                                                    /// --- var-related ---
             LAgs(numlist int >0 sort)              /// 
             EXog(varlist ts)                       /// 
             noCONStant                             ///
             dfk                                    ///  
             SMall                                  /// 
                                                    /// --- output-related ---
             Level(cilevel)                         /// 
             var                                    /// 
             noCNSReport                            /// do not display constraints
             noTable                                /// do not display coefficient table
                                                    ///
                                                          /// --- max-related ---
             noLOg                                        /// -nolog- and -from- not parsed by -mlopts- ! but help is still under [R] maximize
             from(string)                                 /// 
             EVALMode(numlist int min=1 max=1 >=0 <=2)    /// choice 0-2, corresponding to evalmode d0, d1, d2
             GLSIter(numlist int min=1 max=1 >=0) /// 
             STOLerance(real 1e-4)                  /// convergence tolerance for res-covmats
             BTOLerance(real 1e-4)                  /// convergence tolerance for svarih e(b)
             GLSTRace                               /// for each iter, shows -svarih- parameter vector, red-form sigmas, and ML iterations
             fixedfrom                              /// do not start ML optimization w/ estimates from previous GLS iteration
                                                        ///
                                                    /// --- auxiliary ---
             mlv(string)                            /// version under which -ml- is run; >=11.2
             CONSTraints(numlist)                   /// needed only for parsing: see note in svar.ado
                                                    ///
                                                    /// --- debug-related --- (not visible to user)
             debugkeepres                           /// keeps residuals from var
             debugreturn                            /// return additional intermediate matrices etc.
             debugml(string)                        /// string can be:
                                                    ///   > "check"  for -ml check-
             dxdebug                                /// executes evaluator under debug method, e.g. d1debug
                                                       /*  
          */ * ]                                    // mlopts 


    // MAKING SURE GLOBALS ARE EMPTY
    macro drop T_svarih_bac_*
    mata: _svarih_bac_delexternals()

    // CHECKING DEBUG OPTIONS
    if `"`debugml'"'!="" {
        if !inlist(`"`debugml'"', "check") {
            disp as error `"arg to option -debugml- not recognized"'
            exit 198
        }
    }

    // SVARIH_BAC INPUT/OPTIONS

    if "`dxdebug'"!="" local dxdebug debug
    if "`evalmode'"=="" {
        local evalmode d2`dxdebug'
    }
    else {
        local evalmode d`evalmode'`dxdebug'
    }
    if "`evalmode'"=="d0debug" {
        disp as error `"Option 'dxdebug' cannot be used with ml method d0."'
        exit 198
    }
    local evalname d2

    local cmdline : copy local 0
    local cmdline : subinstr local cmdline " mlv(`mlv')" ""  // -mlv()- was not typed by the user but supplied by -svarih-

    if "`glsiter'"=="" local glsiter 0

    if `stolerance'<=0 | `btolerance'<=0 {
        disp as error "Convergence tolerance criteria must be greater than 0"
        exit 198
    }

    // if -constraints- option weren't there, it would be included in local -options- if the user (incorrectly) made 
    // use of -constraints- ; it would then be parsed by -mlopts-
    _get_diopts diopts options, `options'
    mlopts mlopts, `options'    // at this point all options that are not allowed by -svarih_bac- are caught
    if `"`s(collinear)'"' != "" {
        di as err "option collinear not allowed"
        exit 198
    }

    if "`constraints'" != "" {
        di as err "constraints() not allowed"
        exit 198
    }   

    // REGIME INPUT CHECKS

    local neqs: word count `varlist'
    
    confirm matrix `rgmmat'
    capture assert colsof(`rgmmat')==`=`neqs'+1'
    if _rc {
        disp as error `"# of columns of input matrix `rgmmat' must correspond to 1 + # of endogenous variables"'
        exit 198
    }
    local rgmmatnames : colnames `rgmmat'


    local rgmreq rgmcode `varlist'
    local cmp: list rgmreq == rgmmatnames  // with "==" order matters, with "===" it does not
    if `cmp'!=1 {
        local cmp: list rgmreq === rgmmatnames
        if `cmp'!=1 {
            disp as error `"Column names of input arg rgmmat do not correspond to the list of endogenous variables."'
            exit 198
        }
        else { // rgmmat cols have to be reordered to correspond with endog vars
            disp as error `"Column names of input arg rgmmat have a different order than the endogenous variables."'
            exit 198
        }
    }

    tempname col1dup rgmmat_zeroone coldup nonzeroone
    mata: st_numscalar("`col1dup'", rows(uniqrows(st_matrix("`rgmmat'")[.,1]))!=rows(st_matrix("`rgmmat'")[.,1]) )
    if `col1dup'==1 {
        disp as error `"Column 1 of regime matrix contains duplicate entries."'
        exit 198
    }
    matrix `rgmmat_zeroone' = `rgmmat'[.,2...]
    mata : st_numscalar("`coldup'", rows(uniqrows(st_matrix("`rgmmat_zeroone'")))!=rows(st_matrix("`rgmmat_zeroone'")) )
    if `coldup'==1 {
        disp as error `"Regime matrix contains duplicate patterns of volatilities."'
        exit 198
    }
    local numrows = rowsof(`rgmmat')
    mata : st_numscalar("`nonzeroone'", ( st_matrix("`rgmmat_zeroone'"):==J(`numrows',`neqs',0) :| st_matrix("`rgmmat_zeroone'"):==J(`numrows',`neqs',1) ) != J(`numrows',`neqs',1) )
    if `nonzeroone'==1 {
        disp as error `"Volatility states may only be denoted by zeroes and ones in regime matrix (elements of column 2 and to the right of it)."'
        exit 198
    }
    capture assert matmissing(`rgmmat')==0
    if _rc {
        disp as error `"Input matrix `rgmmat' contains missing values"'
        exit 504
    }
    capture assert rowsof(`rgmmat')>=2
    if _rc {
        disp as error `"Input matrix `rgmmat' must have at least 2 rows"'
        exit 198
    }
    
    tempname col1mat
    matrix `col1mat' = `rgmmat'[1...,1]
    local numrows_rgmmat = rowsof(`col1mat')
    forvalues i=1/`numrows_rgmmat' {
        local col1scalar = `=`col1mat'[`i',1]'
        capture confirm integer number `col1scalar'
        if _rc!=0 | `col1scalar'<0 {
            disp as error `"Column 1 of regime matrix contains non-negative or non-integer value(s)."'
            exit 126
        }
    }
    
    capture confirm numeric variable `rgmvar'
    if _rc {
        disp as error `"Variable '`rgmvar'' must be numeric"'
        exit 108
    }
    
    capture assert `rgmvar'==int(`rgmvar')
    if _rc {
        disp as error `"Regime variable contains non-integer values."'
        exit 126
    }

    qui count if `rgmvar'<0
    if `r(N)'>0 {
        disp as error `"Regime variable contains negative values."'
        exit 411
    }

    // MARK SAMPLE, COLLINEAR AND EXOGENOUS VARIABLES; 

    marksample touse
    markout `touse' `rgmvar'

    if "`exog'" != "" {   // note: exog has been unabbreviated by -syntax-
        markout `touse' `exog'
    }   

    // collinearity
    _rmcoll `varlist' if `touse', `constant'
    local varlist `r(varlist)'

    if "`lags'" == "" {
        local lags 1 2
    }
    else {  // numlist has already been expanded by -syntax-
        if "`: list dups lags'"!="" {
            disp as error `"Option 'lags' may not contain duplicate elements."'
            exit 198
        }
    }

    markout `touse' L(`lags').(`varlist')

    local nlags : word count `lags'
    local mlag  : word `nlags' of `lags'

    qui count if `touse'
    local N = r(N)

    if `mlag' > r(N)-1 {
        di as err "you cannot fit a model with `nlags' and `N' observations"
        exit 2001
    }       

    if "`exog'" != "" {
        _rmcoll `exog' if `touse', `constant'
        local exog `r(varlist)'

        tsunab flist : L(0 `lags').(`varlist') `exog'

        _rmcoll `flist' if `touse', `constant'
        local flist2 `r(varlist)'

        local same : list flist == flist2
        if `same' != 1 {
            di as err "{p 0 4}the exogenous variables may not be collinear with the dependent variables, or their lags{p_end}"
            exit 198
        }   
    }

    if "`exog'" != "" {
        local exogopt "exog(`exog')"
    }   

    local nexog : word count `exog'

    // RECHECK REGIMES AFTER SAMPLE HAS BEEN DETERMINED
    // GENERATION OF VALIDREGIMES, RGMMAT_USED, RGMVAR_USED AND REGIME MAPPING
    qui levelsof `rgmvar' if `touse', local(regimes)  // r(levelsof) is always sorted
    local numregimes: word count `regimes'
    if `numregimes'<2 {
        disp as error `"Estimation sample contains fewer than 2 regimes."'
        exit 198
    }

    tempname rgmmat_used rgmmat_used_col1
    tempvar rgmvar_used

    local newrgmcode 1
    local regimemap
    local space ""
    qui gen int `rgmvar_used' = .
    foreach rgm of local regimes {
        local rgmmat_rgmrow 0
        local matched false
        forvalues i=1/`numrows_rgmmat' {
            scalar col1scalar = `rgmmat'[`i',1]
            if `rgm'==col1scalar {
                local matched true
                local rgmmat_rgmrow `i'
                continue, break
            }
        }
        if "`matched'"=="false" {
            disp as error `"Level `rgm' of regime variable not matched by any element of first column of regime matrix."'
            exit 198
        }
        else {  // TODO: regimemap no longer needed ; simplify code
            local regimemap "`space'`regimemap'`space'`rgm' `newrgmcode'"
            local space " "
            qui replace `rgmvar_used' = `newrgmcode' if `rgmvar'==`rgm' & `touse'  // TODO: use -recode- here, should be faster
            local ++newrgmcode
            capture confirm matrix `rgmmat_used'  // TODO: use nullmat() instead
            if _rc==111 {
                matrix `rgmmat_used' = `rgmmat'[`rgmmat_rgmrow',1...]
            }
            else {
                matrix `rgmmat_used' = (`rgmmat_used' \ `rgmmat'[`rgmmat_rgmrow',1...] )
            }
        }
    }
    
    capture assert(mod(`: word count `regimemap'',2)==0)  // TODO: remove this check after certification
    if _rc {
        disp as error `"Regime mapping has an uneven number of elements"'
        exit 9
    }

    matrix `rgmmat_used_col1' = `rgmmat_used'[1...,1]
    matrix `rgmmat_used'      = `rgmmat_used'[1...,2...]


    // PARSE CONSTRAINTS
    if "`acns'" != "" {
        tempname acnsmat
        _svarih_mkmatcns, mat(`acns') name(a) neqs(`neqs') tname(`acnsmat') type(acns())
        local imp_cnsa  `r(svar_cnslist)'   
    }
    if "`bcns'" != "" {
        tempname bcnsmat
        _svarih_mkmatcns, mat(`bcns') name(b) neqs(`neqs') tname(`bcnsmat') type(bcns())
        local imp_cnsb  `r(svar_cnslist)'   
    }
    if "`ecns'" != "" {
        tempname ecnsmat
        _svarih_mkmatcns, mat(`ecns') name(e) neqs(`neqs') tname(`ecnsmat') type(ecns())
        local imp_cnse `r(svar_cnslist)'    
    }

    if "`aeq'" != "" {
        tempname aeqmat
        _svarih_mkmatcns, mat(`aeq')  name(a) neqs(`neqs') tname(`aeqmat')  type(aeq())  eq
        local imp_cnsa `imp_cnsa' `r(svar_cnslist)' 
    }
    if "`beq'" != "" {
        tempname beqmat
        _svarih_mkmatcns, mat(`beq')  name(b) neqs(`neqs') tname(`beqmat')  type(beq())  eq
        local imp_cnsb `imp_cnsb' `r(svar_cnslist)' 
    }
    if "`eeq'" != "" {  // imp_cnse will be redone below if one or more cols of the regimemat sum to zero; corresponding E coefs will be constrained to zero
        tempname eeqmat
        _svarih_mkmatcns, mat(`eeq')  name(e) neqs(`neqs') tname(`eeqmat')  type(eeq())  eq
        local imp_cnse `imp_cnse' `r(svar_cnslist)' 
    }

    foreach curtype in a b e {
        foreach cnsitem in ``curtype'constraints' {
            constraint get `cnsitem'
            if `r(defined)'==0 {
                disp as error `"Constraint `cnsitem' in option '`curtype'constraints()' not defined."'
                exit 412
            }
        }
    }    

    foreach curtype in a b e {
        local cns_`curtype' ``curtype'constraints' `imp_cns`curtype''

        foreach cnsitem of local cns_`curtype' {
            constraint get `cnsitem'
            if "`cns_`curtype'_list'" == "" {
                local cns_`curtype'_list "`r(contents)'"
            }
            else {
                local cns_`curtype'_list "`cns_`curtype'_list':`r(contents)'"
            }   
        }
    }
    
    local fullcns `cns_a' `cns_b' `cns_e'


    disp as text "Performing VAR..."

    if "`var'" == "" {
        local dispvar nodisplay  
    }   
    else {
        local dispvar  
    }

    capture noi var `varlist'  if `touse',              ///
                lags(`lags') `exogopt' `dfk' `constant' ///
                /// `lutstats'
                nobigf                                  /// -irf- needs e(bf), -dsimih- does not
                level(`level') `small'                  /// 
                /// `varconstraints' `islog' `isure' `isiterate' `istolerance'  // irrelevant b/c no varcns on indiv coefs allowed by -svarih-
                `dispvar' `cnsreport'

    if _rc > 0 {
        di as err "{cmd:var} returned error " _rc 
        di as err "check the specification of the underlying VAR"
        exit _rc
    }   

    tempname starts               ///
             sigma b_var V_var    ///
                   b     V     ll ///
             a_est b_est e_est    ///
             Cns
        
    // SAVE SELECTED VAR e()-VALUES
    matrix `b_var'  = e(b)
    matrix `V_var'  = e(V)

    matrix `sigma'  = e(Sigma)

    local depvar      `varlist'
    local exog        `e(exog)'     // some exogvars may have been dropped by -var- b/c of collinearity ; they are no longer contained in `exog'
    local tsfmt      "`e(tsfmt)'"
    local timevar    "`e(timevar)'"
    
    local df_eq_var   = e(df_eq)    // # of coefs per -var- eq ; includes constant and exog 
    local k_var       = e(k)        // total # of coefs in all -var- eqs ; includes constants
    
    local tmax        = e(tmax)
    local tmin        = e(tmin)
    local N_gaps      = e(N_gaps)
    
    if "`small'" != "" local df_r_var = e(df_r)

    // CALCULATING REDUCED-FORM VAR SIGMAS
    // see notes in svarih_llu about this section
    local resvarlist
    tempvar resmiss  // "resmiss": residuals missing
    forvalues i=1/`neqs' {
        tempvar res_eq`i'
        local resvarlist `resvarlist' `res_eq`i''
        qui predict double `res_eq`i'' if e(sample) , residuals eq(#`i')
        if "`debugkeepres'"!="" {
            capture drop res_eq`i'
            qui clonevar res_eq`i' = `res_eq`i''
        }
    }
    qui egen int `resmiss' = rowmiss(`resvarlist')

    tempvar  markervar
    tempname dfkfactor
    local    df `df_eq_var'  // df_eq_var: # of regs per VAR eq, includes constant and exog
    scalar  `dfkfactor' = `N' / (`N'-`df')
    qui gen byte `markervar' = .
    foreach rgm of local regimes {
        tempname sigma_rgm`rgm'
        qui replace `markervar' = (`rgmvar'==`rgm' & `resmiss'==0)
        qui count if `markervar'==1
        local N_rgm`s' `r(N)'
        local regimes_Ns `regimes_Ns' `N_rgm`s''
        mata: st_matrix("`sigma_rgm`rgm''", cross(st_data(.,"`resvarlist'","`markervar'"), st_data(.,"`resvarlist'","`markervar'")) / `N_rgm`s'' )
        if "`dfk'"!="" {
            matrix `sigma_rgm`rgm'' = `sigma_rgm`rgm'' * `dfkfactor'
        }
        if "`debugcovmat'"!="" {
            if "`dfk'"!="" exit 198  // do not use -debugcovmat- with -dfk-
            mata: st_matrix("`sigma_rgm`rgm''", variance(st_data(.,"`resvarlist'","`markervar'")))
            matrix `sigma_rgm`rgm'' = `sigma_rgm`rgm'' * (r(N)-1) / r(N)  // variance() and -corr- divide by N-1, which is inconsistent with the ML estimator of the full sample
        }

        if matmissing(`sigma_rgm`rgm'') {
            disp as error `"Reduced-form covariance matrix for regime `rgm' contains at least one missing."'
            mat li `sigma_rgm`rgm''
            exit 504
        }
        
        local sigmalist `sigmalist' `sigma_rgm`rgm''
    }

    // DEFINING GLOBALS FOR ML EVALUATOR
    
    //      LIKELIHOOD
    global T_svarih_bac_neqs          `neqs'
* global T_svarih_bac_regimes    `regimes'
    global T_svarih_bac_numregimes    `numregimes'
    global T_svarih_bac_regimes_Ns    `regimes_Ns'
    global T_svarih_bac_rgmvar_used   `rgmvar_used'
    global T_svarih_bac_rgmmat_used   `rgmmat_used'
    global T_svarih_bac_resvarlist    `resvarlist'
    global T_svarih_bac_sigmalist     `sigmalist'

    if "`dfk'"!="" global T_svarih_bac_df_eq `df_eq_var'

    //      GRADIENT
    global T_svarih_bac_endogvars     `varlist'

    _svarih_mkpmats_bac, neqs(`neqs')
    local aparms   "`r(aparms)'"
    local bparms   "`r(bparms)'"
    local eparms   "`r(eparms)'"

    if "`from'" == "" {
        // dim of starting values: params of A, B, E
        local base = 3*(`neqs'^2)
        matrix `starts' = J(1,`base',1)
        forvalues i = 1/`base' {
            matrix `starts'[1,`i'] =  `i'/1000  // StataCorp uses 1 + `i'/100
        }
        
        // set diag elems to (0.001+maximum value)
        // -svar- uses a different (simpler) algorithm because it can assume some constraints
        mata: st_local("matmax", strofreal(max(st_matrix("`starts'"))))
        forvalues k=1/3 {
            local pos = (`k'-1)*(`neqs'^2) + 1
            forvalues i=1/`neqs' {
                matrix `starts'[1,`pos'] = `matmax' + 0.001
                local pos = `pos' + `neqs' + 1
            }
        }
        local init  "init(`starts', copy)"
    }
    else {
        local init  "init(`from')"

        `mlv' capture ml model `evalmode' _svarih_bac_`evalname'() `aparms' `bparms' `eparms' if `touse', ///
                          const(`fullcns') max `mlopts' search(off) nolog                                 ///
                          nopreserve `init' iter(0)

        if _rc > 0 {
            di as err "initial values not feasible"
            if "`imp_cnsa'`imp_cnsb'`imp_cnse'" != "" {
                constraint drop `imp_cnsa' `imp_cnsb' `imp_cnse'
            }
            exit _rc
        }
        matrix `starts' = e(b) 
    }

    di as txt "Estimating contemporaneous parameters"

    if "`debugml'"=="check" {
        `mlv' noi ml model `evalmode' _svarih_bac_`evalname'() `aparms' `bparms' `eparms' if `touse', ///
                     const(`fullcns')  `mlopts' `log'                                                 /// search(off) 
                     nopreserve `init'
        ml check
        exit
    }

    // GLS ITERATION

    if `glsiter'>0 {
        
        disp as text "Starting GLS Iteration"

        tempname b_old b_new b_vargls V_vargls mrd_b  // "mrd": mreldif
        foreach rgm of local regimes {
            tempname sigma_rgm`rgm'_old mrd_sigma_rgm`rgm'
            matrix  `sigma_rgm`rgm'_old' = `sigma_rgm`rgm''
            scalar `mrd_sigma_rgm`rgm'' = .
        }
        scalar `mrd_b' = .

        if "`glstrace'"!="" {
            disp as text "{hline}"
            disp as text "GlS iteration `iternum':
            disp as text "Starting values for ML parameter vector:"
            mat li `starts' , noblank

            disp _n "Initial regime-specific residual covariance matrices:"
            foreach rgm of local regimes {
                matlist `sigma_rgm`rgm'' , nonames title(Regime `rgm') noblank
            }
        }
    }
    
    if "`glstrace'"!="" local noi noi

    local iternum 0
    local converged_gls 0
    while (!`converged_gls' & `glsiter'>0 & `iternum'<=`glsiter') {

        if "`glstrace'"!="" disp as text _n "ML estimation: iteration `iternum'"
        `mlv' capture `noi' ml model `evalmode' _svarih_bac_`evalname'() `aparms' `bparms' `eparms' if `touse', ///
                             const(`fullcns') max `mlopts' search(off) `log'                                    ///
                             nopreserve `init'

        if _rc > 0 {
            macro drop T_svarih_bac_*
            mata: _svarih_bac_delexternals()
            if "`imp_cnsa'`imp_cnsb'`imp_cnse'" != "" {
                constraint drop `imp_cnsa' `imp_cnsb' `imp_cnse'
            }
            exit _rc
        }

        matrix `b_new' = e(b)
        local ++iternum

        capture noi mata: _svarih_bac_vargls()  // returns newly calculated `sigma_rgm#', based on VAR-GLS
                                                // also returns `b_vargls' and `V_vargls'
        
        if _rc {
            disp as error _n "GLS estimation failed"
            macro drop T_svarih_bac_*
            mata: _svarih_bac_delexternals()
            if "`imp_cnsa'`imp_cnsb'`imp_cnse'" != "" {
                constraint drop `imp_cnsa' `imp_cnsb' `imp_cnse'
            }
            exit _rc
        }
        if "`debugnogls'"!="" {
            disp mreldif(`b_var', `b_vargls')
            disp mreldif(`V_var', `V_vargls')
            assert mreldif(`b_var', `b_vargls')<1e-10
            assert mreldif(`V_var', `V_vargls')<1e-10
        }

        foreach rgm of local regimes {
            scalar `mrd_sigma_rgm`rgm'' = mreldif(`sigma_rgm`rgm'_old', `sigma_rgm`rgm'')
        }
        if `iternum'>1 scalar `mrd_b' = mreldif(`b_old', `b_new')

        local converged_sigma 1
        foreach rgm of local regimes {
            if `mrd_sigma_rgm`rgm''>`stolerance' local converged_sigma 0
        }
        if `converged_sigma' & ///
           `mrd_b'        <`btolerance' local converged_gls 1

        foreach rgm of local regimes {
            matrix `sigma_rgm`rgm'_old' = `sigma_rgm`rgm''
        }

        matrix `b_old' = e(b)  // -svarih- estimates
        
        if "`fixedfrom'"=="" matrix `starts' = e(b)
        
        if "`glstrace'"!="" {
            disp as text "{hline}"
            disp as text "GlS iteration `iternum':
            disp as text "ML parameter vector passed:"
            mat li e(b) , noblank
            disp as text "mreldif to previous iteration: " `mrd_b' _n
            
            disp as text _n "VAR-GLS calculated regime-specific residual covariance matrices:"
            foreach rgm of local regimes {
                matlist `sigma_rgm`rgm'' , nonames title(Regime `rgm') noblank
                disp as text    "mreldif to previous iteration: " `mrd_sigma_rgm`rgm'' _n
            }
        }
        else {
            disp as text "." _c
        }
        
        if `converged_gls'==1 disp as text _n "GLS convergence achieved"

    }

    // FINAL ML
    disp as text _n "Final ML optimization:"
    `mlv' capture noi ml model `evalmode' _svarih_bac_`evalname'() `aparms' `bparms' `eparms' if `touse', ///
                         const(`fullcns') max `mlopts' search(off) `log'                                  ///
                         nopreserve `init'

    macro drop T_svarih_bac_*
    mata: _svarih_bac_delexternals()

    if _rc > 0 {
        if "`imp_cnsa'`imp_cnsb'`imp_cnse'" != "" {
            constraint drop `imp_cnsa' `imp_cnsb' `imp_cnse'
        }
        exit _rc
    }

    display _n

    matrix `b'         = e(b)
    matrix `V'         = e(V)
    scalar `ll'        = e(ll)
    local rc_ml        = e(rc)
    local ic_ml        = e(ic)
    local converged_ml = e(converged)
    local rank         = e(rank)

    local cn: colfullnames `b'

    // POST b, V
    capture matrix `Cns' = get(Cns)  // need the -capture- since Cns does not exist if no constraints have been specified
    if _rc {  // no constraints specified
        local N_cns = 0
        ereturn post  `b' `V'      , esample(`touse') obs(`N') 
    }
    else {
        local N_cns = e(k)-e(rank)   // # of independent constraints
        ereturn post  `b' `V' `Cns', esample(`touse') obs(`N') 
    }

    // RETURN E-VALUES
    capture confirm matrix `from'
    if !_rc {
        ereturn matrix from = `from', copy
    }
    else {
        if `"`from'"'!="" ereturn local from `"`from'"'
    }
    if `"`mlopts'"'!="" ereturn local mlopts `"`mlopts'"'

    ereturn matrix b_var = `b_var' , copy
    ereturn matrix V_var = `V_var'

    foreach curtype in a b e {
        matrix ``curtype'_est' = J(`neqs', `neqs', 0)
        forvalues j = 1/`neqs' {
            forvalues i = 1/`neqs' {
                matrix ``curtype'_est'[`i',`j'] = _b[`curtype'_`i'_`j':_cons]
            }
        }
        matrix colnames ``curtype'_est' = `depvar'
        matrix rownames ``curtype'_est' = `depvar'
    }

    if "`imp_cnsa'`imp_cnsb'`imp_cnse'" != "" {
        constraint drop `imp_cnsa' `imp_cnsb' `imp_cnse'
    }

    // these values determine how the coefficient table is being displayed
    local k_eq  = 3*(`neqs'^2)
    * local k_aux = (`neqs'^2)
    local k_aux = `k_eq'    // return k_eq=k_aux, and separate output table every `neqs'^2 paras; display is also more compact
                            // -svar- does it the same way

    foreach curtype in a b e {
        if "``curtype'eq'" != "" {
            ereturn matrix `curtype'eq  = ``curtype'eqmat', copy
        }
        if "``curtype'cns'" != "" {
            ereturn matrix `curtype'cns = ``curtype'cnsmat', copy
        }
        foreach cnsitem of local cns_`curtype' {
            ereturn local cns_`curtype' "`cns_`curtype'_list'"
        }
    }
    
    qui mat2mac `rgmmat_used_col1' , col(1)
    local regimes `r(mat2mac)'

    matrix `rgmmat_used' = (`rgmmat_used_col1', `rgmmat_used')
    ereturn matrix rgmmat_used = `rgmmat_used'
    ereturn matrix rgmmat      = `rgmmat' , copy

    ereturn matrix E = `e_est', copy  // TODO: I think I do not need option -copy-
    ereturn matrix B = `b_est', copy
    ereturn matrix A = `a_est', copy

    capture matrix `Cns' = get(Cns)  // have to get Cns again; I believe the -ereturn post- deletes it from memory
    if "`idencheck'"=="" {           // option -noidencheck has not been used
        mata: _svarih_bac_iden(`neqs',`numregimes')  // returns local `idencheck_result'
        if "`idencheck_result'"=="1" ereturn local idencheck "passed"
        if "`idencheck_result'"=="0" ereturn local idencheck "failed"
    }
    else {
        ereturn local idencheck "skipped"
    }

    ereturn matrix Sigma  = `sigma' 
    foreach rgm of local regimes {
        matrix colnames `sigma_rgm`rgm'' = `varlist'
        matrix rownames `sigma_rgm`rgm'' = `varlist'
        ereturn matrix Sigma_rgm`rgm' = `sigma_rgm`rgm''
    }

    if `glsiter'>0 {
        local cnames : colfullnames `b_var'
        matrix colnames `b_vargls' = `cnames'
        ereturn matrix b_vargls = `b_vargls'
        matrix colnames `V_vargls' = `cnames'
        matrix rownames `V_vargls' = `cnames'
        ereturn matrix V_vargls = `V_vargls'
    }

    ereturn scalar rc_ml         = `rc_ml'
    ereturn scalar ic_ml         = `ic_ml'
    ereturn scalar converged_ml  = `converged_ml'
    ereturn scalar converged_gls = `converged_gls'
    ereturn scalar N_cns         = `N_cns'
    ereturn scalar k_dv          = `neqs'          // follow convention
    ereturn scalar k_dv_var      = `neqs'          // follow convention

    ereturn scalar k_eq_var      = `neqs'
    ereturn scalar k_eq          = `k_eq'           // determine how coef table is displayed
    ereturn scalar k_aux         = `k_aux'

    ereturn scalar df_eq_var     = `df_eq_var'
    ereturn scalar k_var         = `k_var'
    
    ereturn scalar mlag          = `mlag' 
    ereturn scalar tmax          = `tmax'
    ereturn scalar tmin          = `tmin'
    ereturn scalar N_gaps        = `N_gaps'
 
    ereturn scalar ll            = `ll'
    
    if "`small'" != "" {
        ereturn scalar df_r_var  = `df_r_var'
        local dfr                = `N'-`rank'
        ereturn scalar df_r      = `dfr'
    }   

    if `glsiter'>0 {
        local glsopts `"`fixedfrom' `glstrace' glsiter(`glsiter')"'
        if "`stolerance'"!="" local glsopts `"`glsopts' stolerance(`stolerance')"'
        if "`btolerance'"!="" local glsopts `"`glsopts' btolerance(`btolerance')"'
        local glsopts : list clean glsopts
        ereturn local glsopts `"`glsopts'"'
        ereturn scalar converged_gls = `converged_gls'
        ereturn scalar ic_gls = `iternum'
    }
    ereturn scalar glsiter = `glsiter'

    ereturn local title    "Heteroskedasticity-identified Structural VAR: Bacchiocchi (2011) ML Framework"
    ereturn local small    `small'
    ereturn local tsfmt   "`tsfmt'"
    ereturn local timevar "`timevar'"

    ereturn scalar numregimes = `numregimes'
    ereturn local  regimes    `regimes'
    ereturn local  regimes_Ns `regimes_Ns'
    ereturn local  rgmvar     `rgmvar'
    ereturn local  rgmmatname `rgmmat'
    
    ereturn local depvar      `depvar' 
    ereturn local exog        `exog'
    ereturn local lags        `lags'
    ereturn local nocons      `noconstant'
    ereturn local dfk_var     `dfk'

    ereturn local predict   svarih_bac_p100
    ereturn local cmdline `"svarih bacchiocchi `cmdline'"'
    ereturn local method    Bacchiocchi
    ereturn local cmd       svarih
    ereturn local version   1.0.0

    ereturn scalar rank = `rank'

    if "`debugreturn'"!="" {
        capture drop rgmvar_used
        gen rgmvar_used = `rgmvar_used'
        ereturn local regimemap `regimemap'
    }

    if e(noisily) { // this may help to speed up the bootstrap a little

        _svarih_Dheadernew, `cnsreport'

        if "`table'"!="notable" {
            if "`e(small)'" != "" {
                _svarih_ETable2 , level(`level') dfr(`dfr') `diopts'
            }
            else {
                _svarih_ETable2 , level(`level')  `diopts'
            }
        }
    }
    
end

*** --------------------------------- SUBROUTINES -----------------------------------------

program define _svarih_mkpmats_bac , rclass
// official Stata's -svar- subroutine _mkpmats, modified to suit -svarih_bac-
    syntax , neqs(numlist max=1 >0)

    forvalues i = 1/`neqs' {
        forvalues j = 1/`neqs' {
            // -DCS-: note that the row index runs faster!!! I will have to account for that in the -ml- evaluator
            local aparms   " `aparms' (a_`j'_`i':) "
            local bparms   " `bparms' (b_`j'_`i':) "
            local eparms   " `eparms' (e_`j'_`i':) "
        }
    }

    return local aparms  "`aparms'"
    return local bparms  "`bparms'"
    return local eparms  "`eparms'"
end

*** --------------------------------- MATA ------------------------------------------------

version 11.2
mata:
mata set matastrict on
void _svarih_bac_vargls() {
// variable naming based on var-gls formula in LL (2008), p.1148

    real scalar neqs,
                numregimes,
                numzvars,
                N,
                dfkfactor,
                t,
                s

    real rowvector regimes
    
    real colvector rgmvar,
                   Y,
                   Zysig
                   

    real matrix A,
                Ai,
                B,
                E,
                rgmmat,
                Z,
                ZZsig,
                Cvec,
                Cmat,
                Cvnc,   // covariance matrix
                idx,
                AiBEs
                
    pointer(real colvector) colvector pYs

    pointer(real matrix) colvector pZs,
                                   pAiBEsSigmaInv,
                                   pSigma_rgm,
                                   pv
                
                                   

    neqs        = cols(tokens(st_local("varlist")))
    regimes     = strtoreal(tokens(st_local("regimes")))
    numregimes  = cols(regimes)
    numzvars    = neqs * cols(tokens(st_local("lags"))) + strtoreal(st_local("nexog"))
    N           = colsum(st_data(., st_local("touse")))
    dfkfactor   = st_numscalar(st_local("dfkfactor"))

    A = rowshape( st_matrix("e(b)")[1, (         1)..(  neqs^2)] , neqs)'
    B = rowshape( st_matrix("e(b)")[1, (  neqs^2+1)..(2*neqs^2)] , neqs)'
    E = rowshape( st_matrix("e(b)")[1, (2*neqs^2+1)..(3*neqs^2)] , neqs)'
    rgmmat = st_matrix(st_local("rgmmat"))

    if (rank(A)!=cols(A)) {
        _error(198 , "GLS iteration: matrix A not invertible.")
    }
    Ai = luinv(A)

    pYs            = J(numregimes,1,NULL)
    pZs            = J(numregimes,1,NULL)
    pAiBEsSigmaInv = J(numregimes,1,NULL)
    pSigma_rgm     = J(numregimes,1,NULL)
    pv             = J(numregimes,1,NULL)

    for (s=1;s<=numregimes;s++) {
        AiBEs = Ai * ( B + E * diag(select(rgmmat[.,2..cols(rgmmat)] , rgmmat[.,1]:==regimes[s])) )
        pAiBEsSigmaInv[s] = &( luinv(AiBEs * AiBEs') )
    }

    rgmvar = st_data(., st_local("rgmvar"), st_local("touse"))
    Y      = st_data(., st_local("varlist"), st_local("touse"))
    Z      = st_data(., "L(" + st_local("lags") + ").(" + st_local("varlist") + ") " + st_local("exog"), st_local("touse"))
                    // ordered by variable, within variable ordered by lag
                    // `exog' is parsed by -syntax- to contain variables and ts-ops in expanded form ; still, it does not allow using '*'

    if (st_local("constant")=="") { // option -noconstant- has not been used
        Z = (Z, J(N,1,1))
        numzvars++
    }
    assert(numzvars==strtoreal(st_local("df_eq_var")))  // TODO: remove after testing

    ZZsig = J(numzvars*neqs, numzvars*neqs, 0)
    Zysig = J(numzvars*neqs, 1, 0)

    if (st_local("debugnogls")!="") {
        pAiBEsSigmaInv = &( luinv(st_matrix(st_local("sigma"))) )  // calc Cvnc on the basis of e(Sigma) from -var- ; incorporates dfk
    }

    /*
    // these are the formulas as transcribed from the LL paper
    for (t=1; t<=N; t++) {
        if (rgmvar[t]==1) {
            ZZsig = ZZsig + ((Z[t,.]'*Z[t,.]) # BBsigmaInv)
            Zysig = Zysig + (Z[t,.]' # BBsigmaInv) * Y[t,.]'
        }
        else {
            ZZsig = ZZsig + ((Z[t,.]'*Z[t,.]) # BLBsigmaInv)
            Zysig = Zysig + (Z[t,.]' # BLBsigmaInv) * Y[t,.]'
        }
    }
    */

    for (s=1;s<=numregimes;s++) {
        pZs[s] = &( select(Z, rgmvar:==regimes[s]) )
        pYs[s] = &( select(Y, rgmvar:==regimes[s]) )
    }

    // rewritten formulas for more efficient calculations ; they have been tested to produce identical results
    for (s=1;s<=numregimes;s++) {
        ZZsig = ZZsig + ( cross( *(pZs[s]) , *(pZs[s]) ) # (*(pAiBEsSigmaInv[s])) )
        Zysig = Zysig + vec((*(pAiBEsSigmaInv[s])) * (*(pYs[s]))' * (*(pZs[s])) )
    }

    Cvnc = luinv(ZZsig)
    
    if (hasmissing(Cvnc)) _error(504)
    
    Cvec = (Cvnc * Zysig)'                // Cvec is (n*k x 1), but with coefs having the faster index rather than eqs => need to invert the index order to get b_glsvar
    Cmat = colshape(Cvnc * Zysig, neqs)'  // note the prime ; Cmat has coefs as `b_var', but with each row of C == one VAR equation
    
    if (hasmissing(Cmat)) _error(504)
    for (s=1;s<=numregimes;s++) {
        pv[s] = &( select(Y-Z*Cmat', rgmvar:==regimes[s]) )  // TODO: make more efficient
    }

    idx = vec(colshape((1..numzvars*neqs), neqs))  // e.g. gives vector (1, 3, 5, 2, 4, 6) to make coef index run faster than the eq index
    Cvec = Cvec[idx]
    Cvnc = Cvnc[idx, idx]  // note: no division through N ; see LUT 2005, p.77, eq (3.2.21)

    for (s=1;s<=numregimes;s++) {
        pSigma_rgm[s] = &( ( (*(pv[s])') * (*(pv[s]) ) ) / colsum(rgmvar:==regimes[s]) )
    }

    if (st_local("dfk")=="") {
        for (s=1;s<=numregimes;s++) {
            st_matrix(st_local("sigma_rgm" + strofreal(regimes[s]) ), *(pSigma_rgm[s]))
        }
    }
    else {
        for (s=1;s<=numregimes;s++) {
            st_matrix(st_local("sigma_rgm" + strofreal(regimes[s]) ), *(pSigma_rgm[s]) * dfkfactor)
        }
    }
    
    st_matrix(st_local("V_vargls") , Cvnc )
    st_matrix(st_local("b_vargls") , Cvec )
}

end


*** --------------------------------- MATA ------------------------------------------------

version 11.2
mata:
mata set matastrict on
    void _svarih_bac_delexternals() {
        
        real scalar i
        
        string rowvector globvars
        
        globvars = ("T_svarih_bac_Ostarhat","T_svarih_bac_Kgs","T_svarih_bac_KHA")
        
        for (i=1;i<=cols(globvars);i++) {
            if (findexternal(globvars[i])!=NULL) {
                rmexternal(globvars[i])
            }
        }
        
    }
end

mata:
mata set matastrict on
    void _svarih_bac_iden(n,sbar) {
        A = st_matrix("e(A)")
        B = st_matrix("e(B)")
        E = st_matrix("e(E)")

        nsbar = sbar*n
        Kns = Kmatrix(n*sbar,n*sbar) 
        K_n_s = Kmatrix(n,sbar) 
        Nns =  0.5*(I(n^2*sbar^2)+Kns)
        Hbar = ( ((I(sbar) # K_n_s)) * ((vec(I(sbar))#I(n))) )  # I(n)

        Astar = I(sbar) # A
        AstarInv = luinv(Astar)
        Bstar = I(sbar) # B
        Estar = I(sbar) # E

        rgmmat_used = st_matrix("e(rgmmat_used)")
        rgmmat_used = rgmmat_used[.,2..cols(rgmmat_used)]
        D = diag(rowshape(rgmmat_used,1))

        Q1 = (AstarInv*(Bstar+Estar*D)*(Bstar+Estar*D)'*AstarInv') # AstarInv
        Q2 = (AstarInv*(Bstar+Estar*D)) # AstarInv
        Q3 = (AstarInv*(Bstar+Estar*D)*D) # AstarInv

        rankmat = (-2*Nns*Q1*Hbar , 2*Nns*Q2*Hbar , 2*Nns*Q3*Hbar)

        // note that for some reason I cannot access e(Cns) at this point; but I can use `Cns' that had been defined before
        Cns = st_matrix(st_local("Cns"))

        if (Cns==J(0,0,.)) {  // no constraints specified
            rankmat_full = rankmat
            if (rank(rankmat_full)==3*n^2) {
                // "Check for local identification: passed"  // info now in -svarih- output table
                return
            }
            else {
                if (st_local("idendetail")=="idendetail") {
                    "Check for local identification: failed"
                    "rank expected for identification: " + strofreal(3*n^2)
                    "rank of full matrix: " + strofreal(rank(rankmat_full))
                    "rank of A-block :" + strofreal(rank(rankmat_full[.,1..n^2]))
                    "rank of B-block: " + strofreal(rank(rankmat_full[.,n^2+1..2*n^2]))
                    "rank of E-block: " + strofreal(rank(rankmat_full[.,2*n^2+1..3*n^2]))
                    "rank of B- and E-block (relevant result if A=I(n)): " + strofreal(rank(rankmat_full[.,1*n^2+1..3*n^2]))
                    "# of columns per block (=rank needed per block): " + strofreal(n^2)
                }
            }        
        }
        else {  // constraints exist
            Cns = Cns[.,1..cols(Cns)-1]
            rankmat_full = (rankmat \ Cns)

            if (rank(rankmat_full)==3*n^2) {
                // "Check for local identification: passed"
            }
            else {
                if (st_local("idendetail")=="idendetail") {
                    "Check for local identification: failed"
                    "rank expected for identification: " + strofreal(3*n^2)
                    "rank of full matrix:"
                    "  with contraints:     " + strofreal(rank(rankmat_full))
                    "  without constraints: " + strofreal(rank(rankmat))
                    "rank of A-block:"
                    "  with contraints:     " + strofreal(rank(rankmat_full[.,1..n^2]))
                    "  without constraints: " + strofreal(rank(rankmat[.,1..n^2]))
                    "rank of B-block:"
                    "  with contraints:     " + strofreal(rank(rankmat_full[.,n^2+1..2*n^2]))
                    "  without constraints: " + strofreal(rank(rankmat[.,n^2+1..2*n^2]))
                    "rank of E-block:"
                    "  with contraints:     " + strofreal(rank(rankmat_full[.,2*n^2+1..3*n^2]))
                    "  without constraints: " + strofreal(rank(rankmat[.,2*n^2+1..3*n^2]))
                    "rank of B- and E-block, (relevant result if A=I(n)):"
                    "  with contraints:     " + strofreal(rank(rankmat_full[.,1*n^2+1..3*n^2]))
                    "  without constraints: " + strofreal(rank(rankmat[.,1*n^2+1..3*n^2]))
                    "# of columns per block (=rank needed per block): " + strofreal(n^2)
                    ""
                }
            }        
        }
        st_local("idencheck_result", strofreal(rank(rankmat_full)==3*n^2))
    }
end





program define svarih_bfa, nclass

    version 11.2

    if replay() {
        if "`e(cmd)'" != "svarih" | "`e(method)'"!="BFanelli" error 301

        if _by() {
            error 190 
        }

        syntax [ , Level(cilevel) Full noTable noCNSReport *]
        
        _get_diopts options, `options'
        _svarih_Dheadernew, `cnsreport'
        
        if "`table'"!="notable" {
            if "`e(small)'" != "" {
                local dfr = e(df_r)
                _svarih_ETable2, level(`level') dfr(`dfr') `options'
            }
            else {
                _svarih_ETable2, level(`level') `options'
            }   
        }
        exit
    } 
    else {
        svarih_bfa_est `0'
    }
end

*** ------------------------ svarih_bfa_est ---------------------------------------------------

program define svarih_bfa_est, eclass sortpreserve

    version 11.2
    
    syntax varlist [if] [in]                    ///
                                                    /// --- svarih_bfa-related ---
           , RGMVar(varname)                         /// variable that identifies the regime periods
           [ BCONstraints(numlist >0 int)        /// 
             BEq(string)                         /// 
             BCns(string)                        /// 
             e2constraints(numlist >0 int)       /// 
             e2eq(string)                        /// 
             e2cns(string)                       /// 
             e3constraints(numlist >0 int)       /// 
             e3eq(string)                        /// 
             e3cns(string)                       /// 
             e4constraints(numlist >0 int)       /// 
             e4eq(string)                        /// 
             e4cns(string)                       /// 
             noIDENcheck                         ///
                                                    /// 
                                                    /// --- var-related ---
             LAgs(numlist int >0 sort)              /// 
             EXog(varlist ts)                       /// 
             noCONStant                             ///
             dfk                                    ///  
             SMall                                  /// 
                                                    ///
                                                    /// --- output-related ---
             Level(cilevel)                         /// 
             var                                    /// 
             noCNSReport                            /// do not display constraints
             noTable                                /// do not display coefficient table
                                                    ///
                                                        /// --- max-related ---
             noLOg                                      /// -nolog- and -from- not parsed by -mlopts- ! but help is still under [R] maximize
             from(string)                               /// 
             EVALMode(numlist int min=1 max=1 >=0 <=2)  /// choice 0-2, corresponding to evalmode d0, d1, d2
                                                        ///
                                                    /// --- auxiliary ---
             mlv(string)                            /// version under which -ml- is run; >=11.2
             CONSTraints(numlist)                   /// needed only for parsing: see note in svar.ado
                                                    ///
                                                    /// --- debug-related --- (not visible to user)
             debugkeepres                           /// keeps residuals from vars
             debugreturn                            /// return additional intermediate matrices etc.
             debugml(string)                        /// string can be:
                                                    ///   > "check"  for -ml check-
             dxdebug                                /// executes evaluator under debug method, e.g. d1debug
                                                       /*  
          */ * ]                                    // mlopts 



    // MAKING SURE GLOBALS ARE EMPTY
    macro drop T_svarih_bfa*  // note: there are no Mata externals

    // CHECKING DEBUG OPTIONS
    if `"`debugml'"'!="" {
        if !inlist(`"`debugml'"', "check") {
            disp as error `"arg to option -debugml- not recognized"'
            exit 9
        }
    }

    // SVARIH_BFA OPTIONS
    if "`dxdebug'"!="" local dxdebug debug
    if "`evalmode'"=="" {
        local evalmode d2`dxdebug'
    }
    else {
        local evalmode d`evalmode'`dxdebug'
    }
    if "`evalmode'"=="d0debug" {
        disp as error `"Option 'dxdebug' cannot be used with ml method d0."'
        exit 198
    }   
    local evalname d2

    local cmdline : copy local 0
    local cmdline : subinstr local cmdline " mlv(`mlv')" ""  // -mlv()- was not typed by the user but supplied by -svarih-

    // if -constraints- option weren't there, it would be included in local -options- if the user (incorrectly) made 
    // use of -constraints- ; it would then be parsed by -mlopts-
    _get_diopts diopts options, `options'
    mlopts mlopts, `options'    // at this point all options that are not allowed by -svarih_bfa- are caught
    if `"`s(collinear)'"' != "" {
        di as err "option collinear not allowed"
        exit 198
    }

    if "`constraints'" != "" {
        di as err "constraints() not allowed"
        exit 198
    }   

    // in contrast to bac and llu, bfa always needs constraints for identification
    if "`bconstraints'`e2constraints'`e3constraints'`e4constraints'`bcns'`e2cns'`e3cns'`e4cns'`beq'`e2eq'`e3eq'`e4eq'" == "" { 
        di as err "no constraints specified"
        exit 198
    }
    
    // MARK SAMPLE
    marksample touse
    markout `touse' `rgmvar'
    
    // REGIME INPUT CHECKS

    local neqs: word count `varlist'
    
    capture confirm numeric variable `rgmvar'
    if _rc {
        disp as error `"Variable '`rgmvar'' must be numeric"'
        exit 108
    }
    capture assert `rgmvar'==int(`rgmvar')
    if _rc {
        disp as error `"Regime variable contains non-integer values."'
        exit 126
    }

    local reqlevels2 1 2
    local reqlevels3 1 2 3
    local reqlevels4 1 2 3 4
    qui levelsof `rgmvar' if `touse' , local(regimes)  // note that local regimes is sorted
    capture assert `: list regimes===reqlevels2' | `: list regimes===reqlevels3' | `: list regimes===reqlevels4'
    if _rc {
        disp as error `"Regime variable `rgmvar' must be a dummy variable with seq values from 1 to 2, 1 to 3, or 1 to 4"'
        exit 125
    }
    
    local numregimes : word count `regimes'

    local e3cnsopts "`e3eq'`e3cns'`e3constraints'"
    local e4cnsopts "`e4eq'`e4cns'`e4constraints'"
    if `numregimes'==2 & "`e3cnsopts'`e4cnsopts'"!="" {
        disp as error `"Sample contains 2 regimes, but constraints for regimes 3 and/or 4 specified."'
        exit 198
    }
    if `numregimes'==3 & "`e4cnsopts'"!="" {
        disp as error `"Sample contains 3 regimes, but constraints for regimes 4 specified."'
        exit 198
    }

    forvalues s=2/`numregimes' {
        local etypes `etypes' e`s'
    }
    local etypes_upper = upper("`etypes'")
    
    // COLLINEAR AND EXOGENOUS VARIABLES; 

    if "`exog'" != "" {
        markout `touse' `exog'
    }   

    // collinearity
    _rmcoll `varlist' if `touse', `constant'  // regime-specific collinearity will be detected in the separate VARs
    local varlist `r(varlist)'

    if "`lags'" == "" {
        local lags 1 2
    }
    else {  // numlist has already been expanded by -syntax-
        if "`: list dups lags'"!="" {
            disp as error `"Option 'lags' may not contain duplicate elements."'
            exit 198
        }
    }

    markout `touse' L(`lags').(`varlist')

    local nlags : word count `lags'
    local mlag  : word `nlags' of `lags'

    foreach rgm of local regimes {
        qui count if `touse' & `rgmvar'==`rgm'
        if `nlags' > r(N)-1 {
            di as err "regime variable level `rgm': you cannot fit a model with `nlags' lags and `r(N)' observations"
            exit 2001
        }       
    }
    
    if "`exog'" != "" {   // TODO: this does not prevent multicollinearity for the 0/1 subsample VARs
        _rmcoll `exog' if `touse', `constant'
        local exog `r(varlist)'

        tsunab flist : L(0 `lags').(`varlist') `exog'

        _rmcoll `flist' if `touse', `constant'
        local flist2 `r(varlist)'

        local same : list flist == flist2
        if `same' != 1 {
            di as err "{p 0 4}the exogenous variables may not be collinear with the dependent variables, or their lags{p_end}"
            exit 198
        }   
    }

    if "`exog'" != "" {
        local exogopt "exog(`exog')"
    }   

    local nexog : word count `exog'
    
    foreach rgm of local regimes {
        qui count if `rgmvar'==`rgm' & `touse'
        local regimes_Ns `regimes_Ns' `r(N)'
    }

    // PARSE CONSTRAINTS
    if "`bcns'" != "" {
        tempname bcnsmat
        _svarih_mkmatcns    , mat(`bcns')    name(b)    neqs(`neqs') tname(`bcnsmat')    type(bcns())
        local imp_cnsb  `r(svar_cnslist)'   
    }
    forvalues i=2/4 {
        if "`e`i'cns'" != "" {
            tempname e`i'cnsmat
            _svarih_mkmatcns, mat(`e`i'cns') name(e`i') neqs(`neqs') tname(`e`i'cnsmat') type(e`i'cns())
            local imp_cnse`i' `r(svar_cnslist)'    
        }
    }
    if "`beq'" != "" {
        tempname beqmat
        _svarih_mkmatcns    , mat(`beq')     name(b)    neqs(`neqs') tname(`beqmat')     type(beq())      eq
        local imp_cnsb `imp_cnsb' `r(svar_cnslist)' 
    }
    forvalues i=2/4 {
        if "`e`i'eq'" != "" {
            tempname e`i'eqmat
            _svarih_mkmatcns, mat(`e`i'eq')  name(e`i') neqs(`neqs') tname(`e`i'eqmat')  type(e`i'eq())   eq
            local imp_cnse`i' `imp_cnse`i'' `r(svar_cnslist)'
        }
    }

    foreach curtype in b `etypes' {
        foreach cnsitem in ``curtype'constraints' {
            constraint get `cnsitem'
            if `r(defined)'==0 {
                disp as error `"Constraint `cnsitem' in option '`curtype'constraints()' not defined."'
                exit 412
            }
        }
    }    

    foreach curtype in b `etypes' {
        local cns_`curtype' ``curtype'constraints' `imp_cns`curtype''

        foreach cnsitem of local cns_`curtype' {
            constraint get `cnsitem'
            if "`cns_`curtype'_list'" == "" {
                local cns_`curtype'_list "`r(contents)'"
            }
            else {
                local cns_`curtype'_list "`cns_`curtype'_list':`r(contents)'"
            }   
        }
    }    

    local fullcns `cns_b' `cns_e2' `cns_e3' `cns_e4'  // fullcns does not contain `cns_e3' if numregimes <3; similarly for `cns_e4'

    disp as text "Performing VARs..."

    if "`var'" == "" {
        local dispvar nodisplay  
    }   
    else {
        local dispvar  
    }

    foreach s of local regimes {
    
        capture noi var `varlist'  if `touse' & `rgmvar'==`s',  ///
                lags(`lags') `exogopt' `dfk' `constant'         ///
                /// `lutstats'
                nobigf                                          /// -irf- needs e(bf), -dsimih- does not
                level(`level') `small'                          ///
                /// `varconstraints' `islog' `isure' `isiterate' `istolerance'  // irrelevant b/c no varcns on indiv coefs allowed by -svarih-
                `dispvar' `cnsreport'

        if _rc > 0 {
            di as err "{cmd:var} returned error for regime `s'" _rc 
            di as err "check the specification of the underlying VAR"
            exit _rc
        }   

        // SAVE SELECTED VAR e()-VALUES
        tempname starts                            ///
                 sigma_var`s' b_var`s' V_var`s'    ///
                              b        V        ll ///
                 b_est e2_est e3_est e4_est        ///
                 Cns
    
        matrix `b_var`s''  = e(b)
        matrix `V_var`s''  = e(V)

        matrix `sigma_var`s''  = e(Sigma)

        local tmax_var`s'    = e(tmax)
        local tmin_var`s'    = e(tmin)
        local N_gaps_var`s'  = e(N_gaps)
        local N_var`s'       = e(N)

        if "`small'" != "" local df_r_var`s' = e(df_r)
    }

    local depvar `varlist'
    local exog   `e(exog)'
    local tsfmt   "`e(tsfmt)'"
    local timevar "`e(timevar)'"

    local df_eq_var   = e(df_eq)   // # of coefs per -var- eq ; includes constant and exog
    local k_var       = e(k)       // total # of coefs in all -var- eqs ; includes constants
    
    // DEFINING GLOBALS FOR ML EVALUATOR

    //      LIKELIHOOD
    global T_svarih_bfa_neqs          `neqs'
    global T_svarih_bfa_regimes_Ns    `regimes_Ns'
    global T_svarih_bfa_sigma_var1    `sigma_var1'        // tempnames of relevant matrices; cannot be fetched through st_local from the evaluator b/c svarih_bfa.ado is not the caller
    global T_svarih_bfa_sigma_var2    `sigma_var2'
    global T_svarih_bfa_sigma_var3    `sigma_var3'
    global T_svarih_bfa_sigma_var4    `sigma_var4'

    //      GRADIENT
    global T_svarih_bfa_endogvars     `varlist'
    
    _svarih_mkpmats_bfa, neqs(`neqs') numregimes(`numregimes')
    local bparms    "`r(bparms)'"
    local e2parms   "`r(e2parms)'"
    local e3parms   "`r(e3parms)'"   // not returned if numregimes<3
    local e4parms   "`r(e4parms)'"   // not returned if numregimes<4

    if "`from'" == "" {
        // dim of starting values: params of B, E
        local base = `numregimes'*(`neqs'^2)
        matrix `starts' = J(1,`base',1)
        forvalues i = 1/`base' {
            matrix `starts'[1,`i'] =  `i'/1000  // StataCorp uses 1 + `i'/100
        }
        
        // set diag elems to (0.001+maximum value)
        // -svar- uses a different (simpler) algorithm because it can assume some constraints
        mata: st_local("matmax", strofreal(max(st_matrix("`starts'"))))
        forvalues k=1/`numregimes' {
            local pos = (`k'-1)*(`neqs'^2) + 1
            forvalues i=1/`neqs' {
                matrix `starts'[1,`pos'] = `matmax' + 0.001
                local pos = `pos' + `neqs' + 1
            }
        }
        local init  "init(`starts', copy)"
    }
    else {
        local init  "init(`from')"

        `mlv' capture ml model `evalmode' _svarih_bfa_`evalname'() `bparms' `e2parms' `e3parms' `e4parms'if `touse', ///
                          const(`fullcns') max `mlopts' search(off)  nolog                                           ///
                          nopreserve `init' iter(0) 

        if _rc > 0 {
            di as err "initial values not feasible"
            if "`imp_cnsb'`imp_cnse2'`imp_cnse3'`imp_cnse4'" != "" {
                constraint drop `imp_cnsb' `imp_cnse2' `imp_cnse3' `imp_cnse4'
            }
            exit _rc
        }
        matrix `starts' = e(b) 
    }

    di as txt "Estimating contemporaneous parameters"

    if "`debugml'"=="check" {
         `mlv' capture noi ml model `evalmode' _svarih_bfa_`evalname'() `bparms' `e2parms' `e3parms' `e4parms' if `touse', ///
                        const(`fullcns') max `mlopts' `log'                                                                /// search(off) 
                        nopreserve `init'
        ml check
        exit
    }

    `mlv' capture noi ml model `evalmode' _svarih_bfa_`evalname'() `bparms' `e2parms' `e3parms' `e4parms' if `touse', ///
                          const(`fullcns') max `mlopts' search(off)  `log'                                            ///
                          nopreserve `init'

    macro drop T_svarih_bfa*

    if _rc > 0 {
        if "`imp_cnsb'`imp_cnse2'`imp_cnse3'`imp_cnse4'" != "" {
            constraint drop `imp_cnsb' `imp_cnse2' `imp_cnse3' `imp_cnse4'
        }
        exit _rc
    }

    display _n

    matrix `b'         = e(b)
    matrix `V'         = e(V)
    scalar `ll'        = e(ll)
    local rc_ml        = e(rc)
    local ic_ml        = e(ic)
    local converged_ml = e(converged)
    local rank         = e(rank)

    local N = e(N)

    local cn: colfullnames `b'

    // POST b, V
    matrix `Cns' = get(Cns)
    local N_cns = e(k)-e(rank)   // # of independent constraints
    ereturn post  `b' `V' `Cns', esample(`touse') obs(`N') 

    // RETURN E-VALUES
    capture confirm matrix `from'
    if !_rc {
        ereturn matrix from = `from', copy
    }
    else {
        if `"`from'"'!="" ereturn local from `"`from'"'
    }
    if `"`mlopts'"'!="" ereturn local mlopts `"`mlopts'"'

    foreach s of local regimes {
        ereturn matrix b_var`s' = `b_var`s''
        ereturn matrix V_var`s' = `V_var`s''
    }

    foreach curtype in b `etypes' {
        matrix ``curtype'_est' = J(`neqs', `neqs', 0)
        forvalues j = 1/`neqs' {
            forvalues i = 1/`neqs' {
                matrix ``curtype'_est'[`i',`j'] = _b[`curtype'_`i'_`j':_cons]
            }
        }
        matrix colnames ``curtype'_est' = `depvar'
        matrix rownames ``curtype'_est' = `depvar'
    }

    if "`imp_cnsb'`imp_cnse2'`imp_cnse3'`imp_cnse4'" != "" {
        constraint drop `imp_cnsb' `imp_cnse2' `imp_cnse3' `imp_cnse4'
    }

    // these values determine how the coefficient table is being displayed
    local k_eq  = `numregimes'*(`neqs'^2)
    local k_aux = `k_eq'

    foreach curtype in b `etypes' {
        if "``curtype'eq'" != "" {
            ereturn matrix `curtype'eq  = ``curtype'eqmat', copy
        }
        if "``curtype'cns'" != "" {
            ereturn matrix `curtype'cns = ``curtype'cnsmat', copy
        }
        foreach cnsitem of local cns_`curtype' {
            ereturn local cns_`curtype' "`cns_`curtype'_list'"
        }
    }

    foreach curtype in B `etypes_upper' {
        ereturn matrix `curtype' = ``=lower("`curtype'")'_est', copy  // TODO: I think I do not need option -copy-
    }

    capture matrix `Cns' = get(Cns)  // have to get Cns again; I believe the -ereturn post- deletes it from memory

    if "`idencheck'"=="" {           // option -noidencheck has not been used
        tempname T a C 
        matcproc `T' `a' `C'
        mata: _svarih_bfa_iden()  // returns local `idencheck_result'
        if "`idencheck_result'"=="1" ereturn local idencheck "passed"
        if "`idencheck_result'"=="0" ereturn local idencheck "failed"
    }
    else {
        ereturn local idencheck "skipped"
    }

    foreach s of local regimes {
        ereturn matrix Sigma_var`s'  = `sigma_var`s'' 
    }

    ereturn scalar rc_ml        = `rc_ml'
    ereturn scalar ic_ml        = `ic_ml'
    ereturn scalar converged_ml = `converged_ml'
    ereturn scalar N_cns        = `N_cns'
    ereturn scalar k_dv         = `neqs'      // follow convention
    ereturn scalar k_dv_var     = `neqs'      // follow convention

    ereturn scalar k_eq_var     = `neqs'
    ereturn scalar k_eq         = `k_eq'
    ereturn scalar k_aux        = `k_aux'

    ereturn scalar df_eq_var    = `df_eq_var'
    ereturn scalar k_var        = `k_var'

    ereturn scalar mlag         = `mlag' 

    foreach s of local regimes {
        ereturn scalar tmax_var`s'     = `tmax_var`s''
        ereturn scalar tmin_var`s'     = `tmin_var`s''
        ereturn scalar N_gaps_var`s'   = `N_gaps_var`s''
        ereturn scalar N_var`s'        = `N_var`s''
    }
    ereturn scalar ll         = `ll'
    
    if "`small'" != "" {
        forvalues s=0/1 {
            ereturn scalar df_r_var`s'        = `df_r_var`s''
        }
        local dfr               = `N'-`rank'
        ereturn scalar df_r     = `dfr'        
    }   

    ereturn local title    "Heteroskedasticity-identified Structural VAR: Bacchiocchi/Fanelli (2012) ML Framework"
    ereturn local small    `small'
    ereturn local tsfmt   "`tsfmt'"
    ereturn local timevar "`timevar'"

    ereturn scalar numregimes = `numregimes'
    ereturn local regimes    `regimes'
    ereturn local regimes_Ns `regimes_Ns'
    ereturn local rgmvar     `rgmvar'

    ereturn local depvar     `depvar' 
    ereturn local exog       `exog'
    ereturn local lags       `lags'
    ereturn local nocons     `noconstant'
    ereturn local dfk_var    `dfk'

    ereturn local predict     svarih_bfa_p100
    ereturn local cmdline   `"svarih bfanelli `cmdline'"'
    ereturn local method      BFanelli
    ereturn local cmd         svarih
    ereturn local version     1.0.0

    ereturn scalar rank = `rank'

    if e(noisily) { // this may help to speed up the bootstrap a little

        _svarih_Dheadernew, `cnsreport'

        if "`table'"!="notable" {
            if "`e(small)'" != "" {
                _svarih_ETable2 , level(`level') dfr(`dfr') `diopts'
            }
            else {
                _svarih_ETable2 , level(`level')  `diopts'
            }
        }
    }
end

*** --------------------------------- SUBROUTINES -----------------------------------------

program define _svarih_mkpmats_bfa , rclass
// official Stata's -svar- subroutine _mkpmats, modified to suit -svarih_bfa-
    syntax , neqs(numlist max=1 >0) numregimes(numlist min=1 max=1 >0 <5 int)

    forvalues i = 1/`neqs' {
        forvalues j = 1/`neqs' {
            // the row index runs faster
            local bparms   " `bparms' (b_`j'_`i':) "
            local e2parms   " `e2parms' (e2_`j'_`i':) "
            if `numregimes'>2 {
                local e3parms   " `e3parms' (e3_`j'_`i':) "
            }
            if `numregimes'>3 {
                local e4parms   " `e4parms' (e4_`j'_`i':) "
            }
        }
    }

    return local bparms  "`bparms'"
    return local e2parms  "`e2parms'"
    return local e3parms  "`e3parms'"
    return local e4parms  "`e4parms'"
end

*** --------------------------------- MATA ------------------------------------------------
version 11.2
mata:
mata set matastrict on
void _svarih_bfa_iden() {

    real scalar neqs, numregimes
    
    real matrix B, E2, E3, E4, BEx, Sstar, D, Dplus, jnk, rankmat
    
    numregimes = strtoreal(st_local("numregimes"))
    neqs       = strtoreal(st_local("neqs"))

    B  = st_matrix("e(B)")
    E2 = st_matrix("e(E2)")
    
    Sstar = (st_matrix(st_local("T"))) // note: Sstar==T; I do not need a transpose (T') since the formulas in [P] -makecns- are in terms of row vectors b and bc
    D = Dmatrix(neqs)
    Dplus = luinv(D'D)*D'

    BEx = ( B , (B+E2) \ (B+E2) , (B+E2) )

    if (numregimes>2) {
        E3 = st_matrix("e(E3)")
        jnk = (J(3,3,1) # (B+E3))
        jnk[|1,1 \ 2*neqs,2*neqs|] = BEx
        BEx = jnk
    }
    if (numregimes>3) {
        E4 = st_matrix("e(E4)")
        jnk = (J(4,4,1) # (B+E4))
        jnk[|1,1 \ 3*neqs,3*neqs|] = BEx
        BEx = jnk
    }

    rankmat = (I(numregimes) # Dplus) * ((BEx # I(neqs)) * Sstar)

    if (rank(rankmat)==cols(rankmat)) {
        st_local("idencheck_result", "1")
    }
    else {
        st_local("idencheck_result", "0")
    }
}

end





program define svarih_llu, nclass

    version 11.2

    if replay() {
        if "`e(cmd)'" != "svarih" | "`e(method)'"!="LLutkepohl" error 301

        if _by() {
            error 190 
        }

        syntax [ , Level(cilevel) Full noTable noCNSReport *]
        
        _get_diopts options, `options'
        _svarih_Dheadernew, `cnsreport'
        
        if "`table'"!="notable" {
            if "`e(small)'" != "" {
                local dfr = e(df_r)
                _svarih_ETable2, level(`level') dfr(`dfr') `options'
            }
            else {
                _svarih_ETable2, level(`level') `options'
            }   
        }
        exit
    } 
    else {
        svarih_llu_est `0'
    }
end

*** ------------------------ svarih_llu_est ---------------------------------------------------

program define svarih_llu_est, eclass sortpreserve

    version 11.2

    syntax varlist [if] [in]                        ///
                                                    /// --- svarih_llu-related ---
           , RGMVar(varname)                        /// variable that identifies the regime periods
           [ BCONstraints(numlist >0 int)           /// 
             BEq(string)                            /// 
             BCns(string)                           /// 
             LCONstraints(numlist >0 int)           /// 
             LEq(string)                            /// 
             LCns(string)                           /// 
                                                    /// --- var-related ---
             LAgs(numlist int >0 sort)              /// 
             EXog(varlist ts)                       /// note: `exog' contains varlist in (ts and unabbreviated) expanded form, but does not allow wildcard '*'
             noCONStant                             ///
             dfk                                    ///  
             SMall                                  /// 
                                                    /// --- output-related ---
             Level(cilevel)                         /// 
             var                                    /// 
             noCNSReport                            /// do not display constraints
             noTable                                /// do not display coefficient table
                                                    ///
                                                        /// --- max-related ---
             noLOg                                      /// -nolog- and -from- not parsed by -mlopts- ! but help is still under [R] maximize
             from(string)                               ///
             EVALMode(numlist int min=1 max=1 >=0 <=2)  /// choice 0-2, corresponding to evalmode d0, d1, d2
             GLSIter(numlist int min=1 max=1 >=0)    /// 
             STOLerance(real 1e-4)                  /// convergence tolerance for res-covmats
             BTOLerance(real 1e-4)                  /// convergence tolerance for svarih e(b)
             GLSTRace                               /// for each iter, shows -svarih- parameter vector, red-form sigmas, and ML iterations
             fixedfrom                              /// do not start ML optimization w/ estimates from previous GLS iteration
                                                    ///
                                                    /// --- auxiliary ---
             mlv(string)                            /// version under which -ml- is run; >=11.2
             CONSTraints(numlist)                   /// needed only for parsing: see note in svar.ado
                                                    ///
                                                    /// --- debug-related --- (not visible to user)
             debugkeepres                           /// keeps residuals from var
             debugcovmat                            /// default calcs regime-specific res-covmats w/o demeaning (as in LLU paper)
                                                    ///   so that the likelihood collapses to the one of -svar-
                                                    ///   -debugcovmat- does demeaning of residuals before calcing covmats ; this option exists b/c
                                                    ///     predicted residuals do not have mean zero in subsamples
             debugnogls                             /// GLS iteration replaces Sigmas by identity matrix, so the first GLS iter should lead to identical ML estimates
             debugml(string)                        /// string can be:
                                                    ///   > "check"  for -ml check-
             dxdebug                                /// executes evaluator under debug method, e.g. d1debug
                                                       /*  
          */ * ]                                    // mlopts 


    // MAKING SURE GLOBALS ARE EMPTY
    macro drop T_svarih_llu_* // note: there are no Mata externals

    // CHECKING DEBUG OPTIONS
    if `"`debugml'"'!="" {
        if !inlist(`"`debugml'"', "check") {
            disp as error `"arg to option -debugml- not recognized"'
            exit 198
        }
    }

    // SVARIH_LLU INPUT/OPTIONS
    if "`dxdebug'"!="" local dxdebug debug
    if "`evalmode'"=="" {
        local evalmode d2`dxdebug'
    }
    else {
        local evalmode d`evalmode'`dxdebug'
    }
    if "`evalmode'"=="d0debug" {
        disp as error `"Option 'dxdebug' cannot be used with ml method d0."'
        exit 198
    }
    local evalname d2

    local cmdline : copy local 0
    local cmdline : subinstr local cmdline " mlv(`mlv')" ""  // -mlv()- was not typed by the user but supplied by -svarih-
    
    if "`glsiter'"=="" local glsiter 0

    if `stolerance'<=0 | `btolerance'<=0 {
        disp as error "Convergence tolerance criteria must be greater than 0"
        exit 198
    }

    // if -constraints- option weren't there, it would be included in local -options- if the user (incorrectly) made 
    // use of -constraints- ; it would then be parsed by -mlopts-
    _get_diopts diopts options, `options'
    mlopts mlopts, `options'    // at this point all options that are not allowed by -svarih_llu- are caught
    if `"`s(collinear)'"' != "" {
        di as err "option collinear not allowed"
        exit 198
    }

    if "`constraints'" != "" {
        di as err "constraints() not allowed"
        exit 198
    }   

    // MARK SAMPLE
    marksample touse
    markout `touse' `rgmvar'

    // REGIME INPUT CHECKS
    local neqs: word count `varlist'
    
    capture confirm numeric variable `rgmvar'
    if _rc {
        disp as error `"Variable '`rgmvar'' must be numeric"'
        exit 108
    }
    
    local reqlevels 1 2
    qui levelsof `rgmvar' if `touse' , local(regimes)  // note that local regimes is sorted
    capture assert `: list regimes===reqlevels'
    if _rc {
        disp as error `"Regime variable `rgmvar' must be a binary dummy variable with integer values of 1 and 2"'
        exit 9
    }
    local numregimes : word count `regimes'
    
    // CHECK CONSTRAINT MATRICES

    foreach curmat in lcns leq {
        if "``curmat''"!="" {
            tempname `curmat'mat `curmat'_sq
            _svarih_PARSEab , mat(``curmat'') name(`curmat') neqs(`neqs') tname(``curmat'mat') /// mat() is name of input matrix definition,
                                                                                               //  name() is name of matrix option, tname() is name of matrix returned
            
            // fill row vector with missings to a sqare matrix so it can be passed to parsing subroutines
            capture matrix ``curmat'_sq' = (``curmat'mat' \ J(`=`neqs'-1' , `neqs' , .) )
            if _rc {
                disp as error `"Matrix `curmat' does not have the correct dimensions."'
                exit 198
            }
        }
    }

    // COLLINEAR AND EXOGENOUS VARIABLES

    if "`exog'" != "" {
        markout `touse' `exog'
    }   

    // collinearity
    _rmcoll `varlist' if `touse', `constant'
    local varlist `r(varlist)'

    if "`lags'" == "" {
        local lags 1 2
    }
    else {  // numlist has already been expanded by -syntax-
        if "`: list dups lags'"!="" {
            disp as error `"Option 'lags' may not contain duplicate elements."'
            exit 198
        }
    }

    markout `touse' L(`lags').(`varlist')

    local nlags : word count `lags'
    local mlag  : word `nlags' of `lags'

    qui count if `touse'
    local N = r(N)

    if `mlag' > r(N)-1 {
        di as err "you cannot fit a model with `nlags' and `N' observations"
        exit 2001
    }       

    if "`exog'" != "" {
        _rmcoll `exog' if `touse', `constant'
        local exog `r(varlist)'

        tsunab flist : L(0 `lags').(`varlist') `exog'

        _rmcoll `flist' if `touse', `constant'
        local flist2 `r(varlist)'

        local same : list flist == flist2
        if `same' != 1 {
            di as err "{p 0 4}the exogenous variables may not be collinear with the dependent variables, or their lags{p_end}"
            exit 198
        }   
    }

    if "`exog'" != "" {
        local exogopt "exog(`exog')"
    }   

    local nexog : word count `exog'
    
    // RECHECK REGIMES AFTER SAMPLE HAS BEEN DETERMINED
    qui levelsof `rgmvar' if `touse', local(regimes)  // r(levelsof) is always sorted
    local numregimes: word count `regimes'
    if `numregimes'!=2 {
        disp as error `"Estimation sample must contain exactly two regimes."'
        exit 198
    }

    // PARSE CONSTRAINTS
    // newly defined matrices will have tempnames of option tname()
    // name() and type() are for error messages only
    tempname jnk
    if "`bcns'" != "" {
        tempname bcnsmat
        _svarih_mkmatcns, mat(`bcns')    name(b) neqs(`neqs') tname(`bcnsmat') type(bcns())
        local imp_cnsb  `r(svar_cnslist)'   
    }
    if "`lcns'" != "" {
        // l-matrices have already been defined above ; must not supply -tname(`lcnsmat')- since the correct matrix has
        //   already been defined above ; the call to _mkmatcns here is to define constraints, and it is based on a square matrix
        //   which should not be returned, since L is a row vector
        _svarih_mkmatcns, mat(`lcns_sq') name(l) neqs(`neqs') tname(`jnk')     type(lcns())
        local imp_cnsl  `r(svar_cnslist)'   
    }
    if "`beq'" != "" {
        tempname beqmat
        _svarih_mkmatcns, mat(`beq')     name(b)  neqs(`neqs') tname(`beqmat')  type(beq())   eq
        local imp_cnsb `imp_cnsb' `r(svar_cnslist)' 
    }
    if "`leq'" != "" {
        _svarih_mkmatcns, mat(`leq_sq')  name(l)  neqs(`neqs') tname(`jnk')     type(leq())   eq
        local imp_cnsl `imp_cnsl' `r(svar_cnslist)' 
    }

    foreach curtype in b l {
        foreach cnsitem in ``curtype'constraints' {
            constraint get `cnsitem'
            if `r(defined)'==0 {
                disp as error `"Constraint `cnsitem' in option '`curtype'constraints()' not defined."'
                exit 412
            }
        }
    }    

    foreach curtype in b l {
        local cns_`curtype' ``curtype'constraints' `imp_cns`curtype''

        foreach cnsitem of local cns_`curtype' {
            constraint get `cnsitem'
            if "`cns_`curtype'_list'" == "" {
                local cns_`curtype'_list "`r(contents)'"
            }
            else {
                local cns_`curtype'_list "`cns_`curtype'_list':`r(contents)'"
            }   
        }
    }    
    
    local fullcns `cns_b' `cns_l'

    disp as text "Performing VAR..."

        if "`var'" == "" {
            local dispvar nodisplay  
        }   
        else {
            local dispvar  
        }

        capture noi var `varlist'  if `touse',              ///
                    lags(`lags') `exogopt' `dfk' `constant' ///
                    /// `lutstats'
                    nobigf                                  /// -irf- needs e(bf), -dsimih- does not
                    level(`level') `small'                  ///
                    /// `varconstraints' `islog' `isure' `isiterate' `istolerance'  // irrelevant b/c no varcns on indiv coefs allowed by -svarih-
                    `dispvar' `cnsreport'

        if _rc > 0 {
            di as err "{cmd:var} returned error " _rc 
            di as err "check the specification of the underlying VAR"
            exit _rc
        }   

    tempname starts               ///
             sigma b_var V_var    ///
                   b     V     ll ///
             b_est l_est          ///
             Cns
        
    // SAVE SELECTED VAR e()-VALUES
    matrix `b_var'  = e(b)
    matrix `V_var'  = e(V)

    matrix `sigma'  = e(Sigma)

    local depvar     `varlist'
    local exog       `e(exog)'     // some exogvars may have been dropped by -var- b/c of collinearity ; they are no longer contained in `exog'
    local tsfmt     "`e(tsfmt)'"
    local timevar   "`e(timevar)'"
    
    local neqs        = e(neqs)
    local df_eq_var   = e(df_eq)
    local k_var       = e(k)
    
    local tmax        = e(tmax)
    local tmin        = e(tmin)
    local N_gaps      = e(N_gaps)

    if "`small'" != "" local df_r_var = e(df_r)

    // CALCULATING REDUCED-FORM VAR SIGMAS
    local resvarlist
    tempvar resmiss  // "resmiss": residuals missing
    forvalues i=1/`neqs' {
        tempvar res_eq`i'
        local resvarlist `resvarlist' `res_eq`i''
        qui predict double `res_eq`i'' if e(sample) , residuals eq(#`i')
        if "`debugkeepres'"!="" {
            capture drop res_eq`i'
            qui clonevar res_eq`i' = `res_eq`i''
        }
    }
    qui egen int `resmiss' = rowmiss(`resvarlist')


    tempvar  markervar
    tempname dfkfactor
    local    df `df_eq_var'  // df_eq_var: # of regs per VAR eq, includes constant and exog
    scalar  `dfkfactor' = `N' / (`N'-`df')
    qui gen byte `markervar' = .
    foreach rgm of local regimes {

        tempname sigma_rgm`rgm'
        qui replace `markervar' = (`rgmvar'==`rgm' & `resmiss'==0)

        // see note above about resmeans
        * mata: st_matrix("`sigma_rgm`rgm''", crossdev(st_data(.,"`resvarlist'","`markervar'"), st_matrix("`resmeans'"), st_data(.,"`resvarlist'","`markervar'"), st_matrix("`resmeans'"))/`r(N)')
        qui count if `markervar'==1
        local N_rgm`s' `r(N)'
        local regimes_Ns `regimes_Ns' `N_rgm`s''
        mata: st_matrix("`sigma_rgm`rgm''", cross(st_data(.,"`resvarlist'","`markervar'"), st_data(.,"`resvarlist'","`markervar'")) / `N_rgm`s'' )
        if "`dfk'"!="" {
            // with -dfk-, -var-'s e(Sigma) is N-df, where df = avg # of regressors per eq
            //             here sigma_rgm1 is div by (N-df)*N1/N = N1*(N-df)/N = N1-df*N1/T
            matrix `sigma_rgm`rgm'' = `sigma_rgm`rgm'' * `dfkfactor'
        }
        if "`debugcovmat'"!="" {
            if "`dfk'"!="" exit 198  // do not use -debugcovmat- with -dfk-
            mata: st_matrix("`sigma_rgm`rgm''", variance(st_data(.,"`resvarlist'","`markervar'")))
            * qui corr `resvarlist' if `markervar' , covariance  // using -corr- gives the same result as mata's variance()
            * matrix `sigma_rgm`rgm'' = r(C)
            matrix `sigma_rgm`rgm'' = `sigma_rgm`rgm'' * (r(N)-1) / r(N)  // variance() and -corr- divide by T-1, which is inconsistent with the ML estimator of the full sample
        }

        if matmissing(`sigma_rgm`rgm'') {
            disp as error `"Reduced-form covariance matrix for regime `rgm' contains at least one missing."'
            mat li `sigma_rgm`rgm''
            exit 504
        }
    }

    // ML / GLS

    //      DEFINING GLOBALS FOR ML EVALUATOR

    //          LIKELIHOOD
    global T_svarih_llu_neqs             `neqs'
    global T_svarih_llu_numregimes       `numregimes'
    global T_svarih_llu_regimes_Ns       `regimes_Ns'
    global T_svarih_llu_sigma_rgm1       `sigma_rgm1'
    global T_svarih_llu_sigma_rgm2       `sigma_rgm2'

    //          GRADIENT
    global T_svarih_llu_endogvars        `varlist'

    //      ML PREP
    _svarih_mkpmats_llu, neqs(`neqs')
    local bparms   "`r(bparms)'"
    local lparms   "`r(lparms)'"

    if "`from'" == "" {
        // dim of starting values: params of B, lambda
        local base = (`neqs'^2) + `neqs'
        matrix `starts' = J(1,`base',1)
        forvalues i = 1/`base' {
            matrix `starts'[1,`i'] =  `i'/1000  // StataCorp uses 1 + `i'/100
        }
        
        // set diag elems of B to (0.001+maximum value)
        // -svar- uses a different (simpler) algorithm because it can assume some constraints
        mata: st_local("matmax", strofreal(max(st_matrix("`starts'"))))
        forvalues i=1/`neqs' {
            local pos = ((`i'-1) *`neqs') + `i'
            matrix `starts'[1,  `pos'] = `matmax' + 0.001
        }
        matrix colnames `starts' = `bparms' `lparms'
        local init  "init(`starts', copy)"
    }
    else {
        local init  "init(`from')"

        `mlv' capture ml model `evalmode' _svarih_llu_`evalname'() `bparms' `lparms' if `touse', ///
                         const(`fullcns') max `mlopts' search(off)                               ///
                         nopreserve `init' iter(0) nolog

        if _rc > 0 {
            di as err "initial values not feasible"
            if "`imp_cnsb'`imp_cnsl'" != "" {
                constraint drop `imp_cnsb' `imp_cnsl'
            }
            exit _rc
        }
        matrix `starts' = e(b) 
    }

    di as txt "Estimating contemporaneous parameters"

    if "`debugml'"=="check" {
        `mlv' noi ml model `evalmode' _svarih_llu_`evalname'() `bparms' `lparms' if `touse', ///
                     const(`fullcns')  `mlopts'                                              /// search(off)
                     nopreserve `init' `log'
        ml check
        exit
    }
    
    // GLS ITERATION

    if `glsiter'>0 {
        
        disp as text "Starting GLS Iteration"

        tempname b_old b_new B L BBsigma BLBsigma sigma_rgm1_old sigma_rgm2_old b_vargls V_vargls
        matrix `sigma_rgm1_old' = `sigma_rgm1'
        matrix `sigma_rgm2_old' = `sigma_rgm2'

        tempname mrd_sigma_rgm1 mrd_sigma_rgm2 mrd_b   // "mrd": mreldif
        scalar `mrd_sigma_rgm1' = .    
        scalar `mrd_sigma_rgm2' = .
        scalar `mrd_b'          = .

        if "`glstrace'"!="" {
            disp as text "{hline}"
            disp as text "GlS iteration `iternum':
            disp as text "Starting values for ML parameter vector:"
            mat li `starts' , noblank

            disp _n "Initial regime-specific residual covariance matrices:"
            matlist `sigma_rgm1' , nonames title(Regime 1) noblank
            matlist `sigma_rgm2' , nonames title(Regime 2) noblank
        }
    }
    
    if "`glstrace'"!="" local noi noi

    local iternum 0
    local converged_gls 0
    while (!`converged_gls' & `glsiter'>0 & `iternum'<`glsiter') {

        if "`glstrace'"!="" disp as text _n "ML estimation: iteration `iternum'"
        `mlv' capture `noi' ml model `evalmode' _svarih_llu_`evalname'() `bparms' `lparms' if `touse', ///
                               const(`fullcns') max `mlopts' search(off)                               ///
                               nopreserve `init' `log'

        if _rc > 0 {
            macro drop T_svarih_llu_*
            if "`imp_cnsb'`imp_cnsl'" != "" {
                constraint drop `imp_cnsb' `imp_cnsl'
            }
            exit _rc
        }
        
        matrix `b_new' = e(b)
        local ++iternum

        mata: st_matrix("`B'" , rowshape(st_matrix("e(b)")[1..`=`neqs'^2'] , `neqs')') 
        mata: st_matrix("`L'" , diag(st_matrix("e(b)")[`=`neqs'^2+1'..`=`neqs'^2+`neqs''])) 
        matrix `BBsigma'  = `B'*    `B''
        matrix `BLBsigma' = `B'*`L'*`B''

        capture mata: _svarih_llu_vargls()  // returns newly calculated `sigma_rgm#', based on VAR-GL 
                                            // also returns `b_vargls' and `V_vargls'
        
        if _rc {
            disp as error _n "GLS estimation failed"
            macro drop T_svarih_llu_*
            if "`imp_cnsb'`imp_cnsl'" != "" {
                constraint drop `imp_cnsb' `imp_cnsl'
            }
            exit _rc
        }
        if "`debugnogls'"!="" {
            disp mreldif(`b_var', `b_vargls')
            disp mreldif(`V_var', `V_vargls')
            assert mreldif(`b_var', `b_vargls')<1e-10
            assert mreldif(`V_var', `V_vargls')<1e-10
        }

        scalar `mrd_sigma_rgm1' = mreldif(`sigma_rgm1_old', `sigma_rgm1')
        scalar `mrd_sigma_rgm2' = mreldif(`sigma_rgm2_old', `sigma_rgm2')
        if `iternum'!=1 scalar `mrd_b' = mreldif(`b_old', `b_new')

        if `mrd_sigma_rgm1'<`stolerance' & ///
           `mrd_sigma_rgm2'<`stolerance' & ///
           `mrd_b'        <`btolerance' local converged_gls 1

        matrix `sigma_rgm1_old' = `sigma_rgm1'
        matrix `sigma_rgm2_old' = `sigma_rgm2'
        matrix `b_old' = e(b)  // -svarih- estimates
        
        if "`fixedfrom'"=="" matrix `starts' = e(b)
        
        if "`glstrace'"!="" {
            disp as text "{hline}"
            disp as text "GlS iteration `iternum':
            disp as text "ML parameter vector passed:"
            mat li e(b) , noblank
            disp as text "mreldif to previous iteration: " `mrd_b' _n
            
            disp as text _n "VAR-GLS calculated regime-specific residual covariance matrices:"
            matlist `sigma_rgm1' , nonames title(Regime 1) noblank
            disp as text    "mreldif to previous iteration: " `mrd_sigma_rgm1' _n
            matlist `sigma_rgm2' , nonames title(Regime 2) noblank
            disp as text    "mreldif to previous iteration: " `mrd_sigma_rgm2' _n
        }
        else {
            disp as text "." _c
        }
        
        if `converged_gls'==1 disp as text _n "GLS convergence achieved"

    }

    // FINAL ML
    disp as text _n "Final ML optimization:"
    `mlv' capture noi ml model `evalmode' _svarih_llu_`evalname'() `bparms' `lparms' if `touse', ///
                         const(`fullcns') max `mlopts' search(off)                               ///
                         nopreserve `init' `log'

    macro drop T_svarih_llu_*

    if _rc > 0 {
        if "`imp_cnsb'`imp_cnsl'" != "" {
            constraint drop `imp_cnsb' `imp_cnsl'
        }
        exit _rc
    }

    display _n
    
    matrix `b' = e(b)
    local cn: colfullnames `b'

    matrix `b'          = e(b)
    matrix `V'          = e(V)
    scalar `ll'         = e(ll)
    local rc_ml         = e(rc)
    local ic_ml         = e(ic)
    local converged_ml  = e(converged)
    local rank          = e(rank)

    // POST b, V
    capture matrix `Cns' = get(Cns)  // need the -capture- since Cns does not exist if no constraints have been specified
    if _rc {  // no constraints specified
        local N_cns = 0
        ereturn post  `b' `V'      , esample(`touse') obs(`N') 
    }
    else {
        local N_cns = e(k)-e(rank)   // # of independent constraints
        ereturn post  `b' `V' `Cns', esample(`touse') obs(`N') 
    }

    // WALD IDENT TEST
    // see e.g. Ltkepohl/Netsunajev 2012 WP, pp.11/12
    tempname wald wald_min wald_p
    matrix `wald' = J(`neqs'*(`neqs'-1)/2, 4, .)
    matrix colnames `wald' = L_ii L_jj chi2 p
    local row 1
    forvalues ii=1/`neqs' {
        forvalues jj=1/`neqs' {
            if `jj'>`ii' {
                qui test [l_1_`ii']_cons=[l_1_`jj']_cons
                matrix `wald'[`row',1] = `ii'
                matrix `wald'[`row',2] = `jj'
                matrix `wald'[`row',3] = r(chi2)
                matrix `wald'[`row',4] = r(p)
                local ++row
            }
        }
    }
    mata: st_numscalar("`wald_min'", colmin(st_matrix("`wald'")[.,3]))
    mata: st_numscalar("`wald_p'"  , colmax(st_matrix("`wald'")[.,4]))

    // RETURN E-VALUES
    ereturn scalar Wald_min = `wald_min'
    ereturn scalar Wald_p   = `wald_p'
    ereturn matrix Wald     = `wald'

    capture confirm matrix `from'
    if !_rc {
        ereturn matrix from = `from', copy
    }
    else {
        if `"`from'"'!="" ereturn local from `"`from'"'
    }
    if `"`mlopts'"'!="" ereturn local mlopts `"`mlopts'"'

    ereturn matrix b_var = `b_var' , copy
    ereturn matrix V_var = `V_var'

    matrix `b_est' = J(`neqs', `neqs', 0)
    matrix `l_est' = J( 1    , `neqs', 0)
    forvalues j = 1/`neqs' {
        matrix `l_est'[1, `j'] = _b[l_1_`j':_cons]
        forvalues i = 1/`neqs' {
            matrix `b_est'[`i',`j'] = _b[b_`i'_`j':_cons]
        }
    }
    matrix colnames `b_est' = `depvar'
    matrix rownames `b_est' = `depvar'

    if "`imp_cnsb'`imp_cnsl'" != "" {
        constraint drop `imp_cnsb' `imp_cnsl'
    }

    // these values determine how the coefficient table is being displayed
    local k_eq  = `neqs'^2+`neqs'
    local k_aux = `k_eq'

    foreach curtype in b l {
        if "``curtype'eq'" != "" {
            ereturn matrix `curtype'eq  = ``curtype'eqmat', copy
        }
        if "``curtype'cns'" != "" {
            ereturn matrix `curtype'cns = ``curtype'cnsmat', copy
        }
        foreach cnsitem of local cns_`curtype' {
            ereturn local cns_`curtype' "`cns_`curtype'_list'"
        }
    }

    matrix colnames `l_est' = `: colnames `b_est''
    ereturn matrix L = `l_est', copy  // TODO: I think I do not need option -copy-
    ereturn matrix B = `b_est', copy

    capture matrix `Cns' = get(Cns)  // have to get Cns again; I believe the -ereturn post- deletes it from memory

    ereturn matrix Sigma  = `sigma'
    foreach rgm of local regimes {
        matrix colnames `sigma_rgm`rgm'' = `varlist'
        matrix rownames `sigma_rgm`rgm'' = `varlist'
        ereturn matrix Sigma_rgm`rgm' = `sigma_rgm`rgm''
    }

    if `glsiter'>0 {
        local cnames : colfullnames `b_var'
        matrix colnames `b_vargls' = `cnames'
        ereturn matrix b_vargls = `b_vargls'
        matrix colnames `V_vargls' = `cnames'
        matrix rownames `V_vargls' = `cnames'
        ereturn matrix V_vargls = `V_vargls'
    }

    ereturn scalar rc_ml         = `rc_ml'
    ereturn scalar ic_ml         = `ic_ml'
    ereturn scalar converged_ml  = `converged_ml'
    ereturn scalar converged_gls = `converged_gls'
    ereturn scalar N_cns         = `N_cns'
    ereturn scalar k_dv          = `neqs'          // follow convention
    ereturn scalar k_dv_var      = `neqs'          // follow convention

    ereturn scalar k_eq_var      = `neqs'
    ereturn scalar k_eq          = `k_eq'
    ereturn scalar k_aux         = `k_aux'

    ereturn scalar df_eq_var     = `df_eq_var'
    ereturn scalar k_var         = `k_var'
    
    ereturn scalar mlag          = `mlag' 
    ereturn scalar tmax          = `tmax'
    ereturn scalar tmin          = `tmin'
    ereturn scalar N_gaps        = `N_gaps'

    ereturn scalar ll            = `ll'
    
    if "`small'" != "" {
        ereturn scalar df_r_var  = `df_r_var'
        local dfr                = `N'-`rank'
        ereturn scalar df_r      = `dfr'
    }

    if `glsiter'>0 {
        local glsopts `"`fixedfrom' `glstrace' glsiter(`glsiter')"'
        if "`stolerance'"!="" local glsopts `"`glsopts' stolerance(`stolerance')"'
        if "`btolerance'"!="" local glsopts `"`glsopts' btolerance(`btolerance')"'
        local glsopts : list clean glsopts
        ereturn local glsopts `"`glsopts'"'
        ereturn scalar converged_gls = `converged_gls'
        ereturn scalar ic_gls = `iternum'
    }
    ereturn scalar glsiter = `glsiter'

    ereturn local title   "Heteroskedasticity-identified Structural VAR: Lanne/Ltkepohl (2008) ML Framework"
    ereturn local small    `small'
    ereturn local tsfmt   "`tsfmt'"
    ereturn local timevar "`timevar'"

    ereturn scalar numregimes = `numregimes'
    ereturn local regimes    `regimes'
    ereturn local regimes_Ns `regimes_Ns'
    ereturn local rgmvar     `rgmvar'
    
    ereturn local depvar     `depvar' 
    ereturn local exog       `exog'
    ereturn local lags       `lags'
    ereturn local nocons     `noconstant'
    ereturn local dfk_var    `dfk'

    ereturn local predict   svarih_llu_p100
    ereturn local cmdline `"svarih llutkepohl `cmdline'"'
    ereturn local method    LLutkepohl
    ereturn local cmd       svarih
    ereturn local version   1.0.0

    ereturn scalar rank = `rank'

    if e(noisily) { // this may help to speed up the bootstrap a little

        _svarih_Dheadernew, `cnsreport'

        if "`table'"!="notable" {
            if "`e(small)'" != "" {
                _svarih_ETable2 , level(`level') dfr(`dfr') `diopts'
            }
            else {
                _svarih_ETable2 , level(`level')  `diopts'
            }
        }
    }

end

*** --------------------------------- SUBROUTINES -----------------------------------------

program define _svarih_mkpmats_llu , rclass
// official Stata's -svar- subroutine _mkpmats, modified to suit -svarih_llu-
    syntax , neqs(numlist max=1 >0)

    forvalues i = 1/`neqs' {
        forvalues j = 1/`neqs' {
            // row index runs faster
            local bparms   " `bparms' (b_`j'_`i':) "
            if `j'==1 local lparms " `lparms' (l_`j'_`i':) "
        }
    }

    return local bparms  "`bparms'"
    return local lparms  "`lparms'"

end

*** --------------------------------- MATA ------------------------------------------------

version 11.2
mata:
mata set matastrict on
void _svarih_llu_vargls() {
// variable naming based on var-gls formula in LL (2008), p.1148

    real scalar neqs,
                numzvars,
                N,
                dfkfactor,
                t

    real colvector rgmvar,
                   Y,
                   Y1,
                   Y2,
                   Zysig

    real matrix Z,
                ZZsig,
                Z1,
                Z2,
                BBsigmaInv,
                BLBsigmaInv,
                Cvec,
                Cmat,
                Cvnc,   // covariance matrix
                idx,
                v1,
                v2,
                sigma_rgm1,
                sigma_rgm2


    neqs        = cols(tokens(st_local("varlist")))
    numzvars    = neqs * cols(tokens(st_local("lags"))) + strtoreal(st_local("nexog"))
    N           = colsum(st_data(., st_local("touse")))
    dfkfactor   = st_numscalar(st_local("dfkfactor"))
    BBsigmaInv  = luinv(st_matrix(st_local("BBsigma")))
    BLBsigmaInv = luinv(st_matrix(st_local("BLBsigma")))
    
    rgmvar = st_data(., st_local("rgmvar"), st_local("touse"))
    Y      = st_data(., st_local("varlist"), st_local("touse"))
    Z      = st_data(., "L(" + st_local("lags") + ").(" + st_local("varlist") + ") " + st_local("exog"), st_local("touse"))
                    // ordered by variable, within variable ordered by lag
                    // `exog' is parsed by -syntax- to contain variables and ts-ops in expanded form ; still, it does not allow using '*'

    if (st_local("constant")=="") { // option -noconstant- has not been used
        Z = (Z, J(N,1,1))
        numzvars++
    }
    ZZsig = J(numzvars*neqs, numzvars*neqs, 0)
    Zysig = J(numzvars*neqs, 1, 0)

    if (st_local("debugnogls")!="") {
        BBsigmaInv  = BLBsigmaInv = luinv( st_matrix(st_local("sigma")) )  // calc Cvnc on the basis of e(Sigma) from -var- ; incorporates dfk
    }

    /*
    // these are the formulas as transcribed from the LL paper
    for (t=1; t<=N; t++) {
        if (rgmvar[t]==1) {
            ZZsig = ZZsig + ((Z[t,.]'*Z[t,.]) # BBsigmaInv)
            Zysig = Zysig + (Z[t,.]' # BBsigmaInv) * Y[t,.]'
        }
        else {
            ZZsig = ZZsig + ((Z[t,.]'*Z[t,.]) # BLBsigmaInv)
            Zysig = Zysig + (Z[t,.]' # BLBsigmaInv) * Y[t,.]'
        }
    }
    */

    Z1 = select(Z, rgmvar:==1)
    Z2 = select(Z, rgmvar:==2)
    Y1 = select(Y, rgmvar:==1)
    Y2 = select(Y, rgmvar:==2)

    // rewritten formulas for more efficient calculations ; they have been tested to produce identical results
    ZZsig = (cross(Z1,Z1) # BBsigmaInv) + (cross(Z2,Z2) # BLBsigmaInv)
    Zysig = vec(BBsigmaInv*Y1'*Z1)      + vec(BLBsigmaInv*Y2'*Z2)
    
    Cvnc = luinv(ZZsig)
    
    if (hasmissing(Cvnc)) _error(504)
    
    Cvec = (Cvnc * Zysig)'                // Cvec is (n*k x 1), but with coefs having the faster index rather than eqs => need to invert the index order to get b_glsvar
    Cmat = colshape(Cvnc * Zysig, neqs)'  // note the prime ; Cmat has coefs as `b_var', but with each row of C == one VAR equation
    
    if (hasmissing(Cmat)) _error(504)
    v1 = select(Y-Z*Cmat', rgmvar:==1)
    v2 = select(Y-Z*Cmat', rgmvar:==2)

    idx = vec(colshape((1..numzvars*neqs), neqs))  // e.g. gives vector (1, 3, 5, 2, 4, 6) to make coef index run faster than the eq index
    Cvec = Cvec[idx]
    Cvnc = Cvnc[idx, idx]  // note: no division through N ; see LUT 2005, p.77, eq (3.2.21)

    sigma_rgm1 = (v1'*v1)/colsum(rgmvar:==1)
    sigma_rgm2 = (v2'*v2)/colsum(rgmvar:==2)
    
    if (st_local("dfk")=="") {
        st_matrix(st_local("sigma_rgm1"), sigma_rgm1)
        st_matrix(st_local("sigma_rgm2"), sigma_rgm2)
    }
    else {
        st_matrix(st_local("sigma_rgm1"), sigma_rgm1 * dfkfactor)
        st_matrix(st_local("sigma_rgm2"), sigma_rgm2 * dfkfactor)
    }
    
    st_matrix(st_local("V_vargls") , Cvnc )
    st_matrix(st_local("b_vargls") , Cvec )
}

end




program define svarih_cmat, rclass

    version 11.2

    if `"`e(cmd)'"'!="svarih" error 301

    // parse common options first
    capture syntax , [ STar EXTended noPval Format(string) * ]
    local 0 `", `options'"'

    // parse method-dependent options
    if "`e(method)'"=="Bacchiocchi" {
        syntax , [ a                  ///
                   b                  ///
                   e                  ///
                   BPluse             ///  note that bfa has an option with the same name but that takes a numlist as arg
                 ]
    }
    else if "`e(method)'"=="BFanelli" {
        syntax , [ b                  ///
                   e2                 ///
                   e3                 ///
                   e4                 ///
                   BPluse(numlist min=1 max=3 int >=2 <=4 sort) ///
                 ]
    }
    else if "`e(method)'"=="LLutkepohl" {
        syntax , [ b                  ///
                   l                  ///
                   bl                 ///
                 ]
    }
    else {
        exit 301
    }

    if `"`format'"'!="" {
        capture confirm format `format'
        if _rc error 120
    }

    local mshort  = lower(substr("`e(method)'",1,3))
    local mshortu = upper(substr("`e(method)'",1,3))

    local neqs       `e(k_dv)'
    local endog      `e(depvar)'
    local numregimes `e(numregimes)'

    if "`mshort'"=="bfa" {
        qui numlist "2/`numregimes'"
        local bfa_eregimes `r(numlist)'

        if ("`numregimes'"=="2" & "`e3'`e4'"!="") | ///
           ("`numregimes'"=="3" &     "`e4'"!="") {
                disp as error `"Usage of options 'e3' and/or 'e4' not consistent with current model."'
                exit 198
        }

        if "`bpluse'"!="" {
            local bpluse : list uniq bpluse
            if !`: list bpluse in bfa_eregimes' {
                disp as error `"Invalid regimes # in option 'bpluse'."'
                exit 125
            }
        }
    }

    // GETTING MATRICES FROM e()
    local bacmats A B E
    local bfamats   B   E2 E3 E4
    local llumats   B            L
    local allmats A B E E2 E3 E4 L

    // local matrices have prefix "mat_" and contain upper case letters to disinguish them from options
    //   e.g. option a, matrix mat_A
    foreach mu of local allmats {
        tempname mat_`mu'
        capture matrix `mat_`mu'' = e(`mu')
    }

    // DEFAULT RETURN MATRICES
    if "`a'`b'`e'`bpluse'`e2'`e3'`e4'`l'`bl'"=="" {

        local b      b

        if "`mshort'"=="bac" {
            local a      a
            local e      e
        }
        else if "`mshort'"=="bfa" {
            foreach s of local bfa_eregimes {
                local e`s' e`s'
            }
        }
        else if "`mshort'"=="llu" {
            local l      l
        }
    }

    // CALC SIMPLE MATRICES
    local simplemats `a' `b' `e' `e2' `e3' `e4' `l'

    foreach m of local simplemats {
        
        if "``m''"!="" {

            local mu = upper("`m'")
            local returnlist `returnlist' mat_`mu'

            if "`pval'"=="nopval" {
                matlist `mat_`mu'', format(`format') title(SVARIH `mshortu' matrix `mu':)
            }
            else {
                tempname mat_`mu'p
                matrix `mat_`mu'p' = e(V)
                if "`mu'"=="L" {  // L is a row vector
                    matrix `mat_`mu'p' = `mat_`mu'p'["`m'_1_1:_cons".."`m'_1_`neqs':_cons"     ,"`m'_1_1:_cons".."`m'_1_`neqs':_cons"]
                }
                else {
                    matrix `mat_`mu'p' = `mat_`mu'p'["`m'_1_1:_cons".."`m'_`neqs'_`neqs':_cons","`m'_1_1:_cons".."`m'_`neqs'_`neqs':_cons"]
                }
                mata: st_matrix("`mat_`mu'p'", rowshape(diagonal(st_matrix("`mat_`mu'p'")), `neqs' )' )
                matrix colnames `mat_`mu'p' = `endog'
                if "`mu'"!="L" matrix rownames `mat_`mu'p' = `endog'

                pvalmat, coefmat(`mat_`mu'') varmat(`mat_`mu'p') `star' `extended' // pvalmat accounts for small sample stats when e(df_r) is present
                matrix `mat_`mu''  = r(mergemat)
                matrix `mat_`mu'p' = r(pvalmat)

                matlist `mat_`mu'', format(`format') title(SVARIH `mshortu' matrix `mu': coef & p-values)
            }
        }
    }

    // CALC MATRICES OF LINCOMS OR NLLINCOMS
    if "`bpluse'"!="" {
    
        foreach s of local bpluse {
            
            if "`mshort'"=="bac" local s ""

            local returnlist `returnlist' mat_BplusE`s'
            
            tempname mat_BplusE`s'  mat_BplusE`s'se
            matrix `mat_BplusE`s''      = J(`neqs', `neqs', .)
            matrix `mat_BplusE`s'se'    = J(`neqs', `neqs', .)

            forvalues i=1/`neqs' {
                forvalues j=1/`neqs' {
                    qui lincom [b_`i'_`j']_cons + [e`s'_`i'_`j']_cons
                    matrix `mat_BplusE`s''[`i', `j']   = r(estimate)
                    matrix `mat_BplusE`s'se'[`i', `j'] = r(se)
                }
            }

            foreach curmat in mat_BplusE`s'  mat_BplusE`s'se {
                matrix colnames ``curmat'' = `endog'
                matrix rownames ``curmat'' = `endog'
            }

            if "`pval'"=="nopval" {
                matlist `mat_BplusE`s'', format(`format') title(SVARIH `mshortu' matrix sum B+E`s':)
            }
            else {
                tempname mat_BplusE`s'p
                pvalmat, coefmat(`mat_BplusE`s'') varmat(`mat_BplusE`s'se') se  `star' `extended'
                matrix `mat_BplusE`s''  = r(mergemat)
                matrix `mat_BplusE`s'p' = r(pvalmat)

                matlist `mat_BplusE`s'', format(`format') title(SVARIH `mshortu' matrix sum B+E`s': coef & p-values)
            }
        }
    }

    if "`bl'"!="" {

        local returnlist `returnlist' mat_BL

        tempname mat_BL mat_BLse jnk
        matrix `mat_BL'      = J(`neqs', `neqs', .)
        matrix `mat_BLse'    = J(`neqs', `neqs', .)

        forvalues i=1/`neqs' {
            forvalues j=1/`neqs' {
                qui nlcom [b_`i'_`j']_cons * sqrt( [l_1_`j']_cons )
                matrix `mat_BL'[`i', `j']   = r(b)
                matrix `jnk'                = r(V)
                matrix `mat_BLse'[`i', `j'] = sqrt(`jnk'[1,1])
            }
        }

        foreach curmat in mat_BL  mat_BLse {
            matrix colnames ``curmat'' = `endog'
            matrix rownames ``curmat'' = `endog'
        }

        if "`pval'"=="nopval" {
            matlist `mat_BL', format(`format') title(SVARIH `mshortu' matrix BL:)
        }
        else {
            tempname mat_BLp
            pvalmat, coefmat(`mat_BL') varmat(`mat_BLse') se  `star' `extended'
            matrix `mat_BL'  = r(mergemat)
            matrix `mat_BLp' = r(pvalmat)

            matlist `mat_BL', format(`format') title(SVARIH `mshortu' matrix BL: coef & p-values)
        }
    }

    // RETURNING R-VALUES
    foreach curmat of local returnlist {
        local retmatname `curmat'  // "retnamename": name of returned matrix
        local retmatname: subinstr local curmat "mat_" ""  
        return matrix `retmatname' = ``curmat''
        if "`pval'"!="nopval" {
            return matrix `retmatname'p = ``curmat'p'
        }
    }

end





program define svarih_examples, rclass

    version 11.2

    syntax [anything(name=estname)] , [ Store EReplace DIRectory(string) NOIsily List clear ]
           
    
    local registered bac_svar bac_first bac_unconstr_gls bac_notident bac_constr_gls bac_svarrepl  ///
                     bfa_svarrepl bfa_svar1 bfa_svar2 bfa_unconstr bfa_constr                      ///
                     llu_svar llu_first llu_unconstr_gls llu_constr_gls llu_lcns
    
    if "`list'"!="" {
        if `"`estname'"'!="" {
            disp as error `"Specifiy either {it:estname} or option 'list'."'
            exit 198
        }
        
        disp as text `"Available svarih example estimates:"' _n
        foreach spec of local registered {
            disp as text "  `spec'"
        }
        exit
    }
    
    confirm name `estname'

    if !`: list estname in registered' {
        disp as error `"Estimates name `estname' not registered in 'svarih examples'."'
        exit 198
    }
    
    if "`store'`ereplace'"=="" {
        disp as error `"When specifying {it:estname}, you must complement it by 'store' and/or 'ereplace'."'
        exit 198
    }
    
    mata: ds_pathparts(`"`c(filename)'"')
    if `"`r(root)'"'!="lutkepohl2" {
        if `"`directory'"'!="" {
            use `"`directory'/lutkepohl2"' , `clear'
        }
        else {
            use lutkepohl2 , `clear'
        }
    }

    preserve
    capture {
        qui keep qtr dln_inv dln_inc dln_consump
        qui tsset
        qui datasignature
        assert "`r(datasignature)'"=="92:4(87965):2258512122:3441521482"
    }
    if _rc {
        disp as error `"Relevant data of original data set lutkepohl2 have been modified."'
        exit 9
    }
    restore
    

    // tempnames of matrices
    tempname rgmmat I3 N3 lowtr1 lowtr diag

    capture drop rgmvar
    qui gen byte rgmvar = (qtr>=tq(1974q1)) + 1
    matrix `rgmmat' = (1 , 0 , 0 , 0 \ 2 , 1 , 1 , 1 \ 3 , 1 , 0 , 1 )
    matrix colnames `rgmmat' = rgmcode dln_inv dln_inc dln_consump

    // generic matrix definitions
    matrix `I3'     = I(3)
    matrix `N3'     = J(3,3,0)
    matrix `lowtr1' = (1,0,0\.,1,0\.,.,1)
    matrix `lowtr'  = (.,0,0\.,.,0\.,.,.)
    matrix `diag'   = (.,0,0\0,.,0\0,0,.)
    
    tempname ehold
    _estimates hold `ehold' , nullok
    
    local qui qui
    if "`noisily'"!="" local qui noi
    
    `qui' {
        if "`estname'"=="bac_svar" { // ************************************ BAC *********************************
            svar dln_inv dln_inc dln_consump , aeq(`lowtr1') beq(`diag')
        }
        else if "`estname'"=="bac_first" {
            svarih bac dln_inv dln_inc dln_consump , rgmvar(rgmvar) rgmmat(`rgmmat') aeq((1,0,0\.,1,0\.,.,1)) beq((.,0,0\0,.,0\0,0,.)) eeq((.,0,0\0,.,0\0,0,.))
        }
        else if "`estname'"=="bac_unconstr_gls" {
            /*
            matrix `aeq' = (1,.,. \ ///
                            .,1,0 \ ///
                            .,.,1)

            matrix `eeq' = (.,0,0 \ ///
                            0,0,0 \ ///
                            0,0,0)
            */
            svarih bac dln_inv dln_inc dln_consump , rgmvar(rgmvar) rgmmat(`rgmmat') aeq((1,.,.\.,1,0\.,.,1)) beq((.,0,0\0,.,0\0,0,.)) eeq((.,0,0\0,0,0\0,0,0)) glsiter(100)
        }
        else if "`estname'"=="bac_notident" {
            /*
            matrix `aeq' = (1,.,. \ ///
                            .,1,. \ ///
                            .,.,1)
            matrix `eeq' = (.,0,0 \ ///
                            0,0,0 \ ///
                            0,0,0)
            */
            svarih bac dln_inv dln_inc dln_consump , rgmvar(rgmvar) rgmmat(`rgmmat') aeq((1,.,.\.,1,.\.,.,1)) beq((.,0,0\0,.,0\0,0,.)) eeq((.,0,0\0,0,0\0,0,0)) glsiter(100)
        }
        else if "`estname'"=="bac_constr_gls" {
            /*
            matrix `eeq' = (.,0,0 \ ///
                            0,0,0 \ ///
                            0,0,0)
            */
            svarih bac dln_inv dln_inc dln_consump , rgmvar(rgmvar) rgmmat(`rgmmat') aeq((1,0,0\.,1,0\.,.,1)) beq((.,0,0\0,.,0\0,0,.)) eeq((.,0,0\0,0,0\0,0,0)) glsiter(100)
        }
        else if "`estname'"=="bac_svarrepl" {
            svarih bac dln_inv dln_inc dln_consump , rgmvar(rgmvar) rgmmat(`rgmmat') aeq((1,0,0\.,1,0\.,.,1)) beq((.,0,0\0,.,0\0,0,.)) eeq(J(3,3,0))
        }
        else if "`estname'"=="bfa_svarrepl" { // ************************************ BFA *********************************
            svarih bfa dln_inv dln_inc dln_consump , rgmvar(rgmvar) beq((.,0,0\.,.,0\.,.,.)) e2eq((.,0,0\.,.,0\.,.,.))
        }
        else if "`estname'"=="bfa_svar1" {
            svar dln_inv dln_inc dln_consump if rgmvar==1, aeq(`I3') beq(`lowtr')
        }
        else if "`estname'"=="bfa_svar2" {
            svar dln_inv dln_inc dln_consump if rgmvar==2, aeq(`I3') beq(`lowtr')
        }
        else if "`estname'"=="bfa_unconstr" {
            /*
            matrix `beq'  = (.,.,0 \ ///
                             .,.,0 \ ///
                             .,.,.)
            matrix `e2eq' = (.,0,0 \ ///
                             .,.,0 \ ///
                             0,0,0)
            */
            svarih bfa dln_inv dln_inc dln_consump , rgmvar(rgmvar) beq((.,.,0\.,.,0\.,.,.)) e2eq((.,0,0\.,.,0\0,0,0))
        }
        else if "`estname'"=="bfa_constr" {
            /*
            matrix `beq'  = (.,.,0 \ ///
                             .,.,0 \ ///
                             .,.,.)
            matrix `e2eq' = (.,0,0 \ ///
                             0,0,0 \ ///
                             0,0,0)
            */
            svarih bfa dln_inv dln_inc dln_consump , rgmvar(rgmvar) beq((.,.,0\.,.,0\.,.,.)) e2eq((.,0,0\0,0,0\0,0,0)) 
        }
        else if "`estname'"=="llu_svar" { // ************************************ LLU *********************************
            svar dln_inv dln_inc dln_consump, aeq(`I3') beq(`lowtr')
        }
        else if "`estname'"=="llu_first" {
            svarih llu dln_inv dln_inc dln_consump , rgmvar(rgmvar) beq((.,0,0\.,.,0\.,.,.))

        }
        else if "`estname'"=="llu_unconstr_gls" {
            svarih llu dln_inv dln_inc dln_consump , rgmvar(rgmvar) glsiter(100)
        }
        else if "`estname'"=="llu_constr_gls" {
            svarih llu dln_inv dln_inc dln_consump , rgmvar(rgmvar) glsiter(100) beq((.,0,0\.,.,0\.,.,.))
        }
        else if "`estname'"=="llu_lcns" {
            /*
            matrix `lcns' = (.,5,5)
            */
            svarih llu dln_inv dln_inc dln_consump , rgmvar(rgmvar) glsiter(100) lcns(.,5,5)
        }
    }
    
    disp ""
    if "`store'" !="" {
        est store `estname'  // does a copy of e()
        disp as text `"svarih example estimates `estname' stored in catalogue of estimates"'
    }
    if "`ereplace'"!="" {
        disp as text `"svarih example estimates `estname' stored in e()"'
    }
    else {
        _est unhold `ehold'
    }
    

end




program define mat2mac, rclass

    version 10.1
    syntax anything(name=matin) ,              ///
           [ Row(numlist min=1 max=1 >0 int)     ///
             Column(numlist min=1 max=1 >0 int)  ///
             comma                               ///
           ]

    confirm matrix `matin'

    if `: word count `matin''!=1 {
        disp as error `"You may only supply one {it:matname}."'
        exit 198
    }

    tempname matname
    matrix `matname' = `matin'  // copy matrix to make sure ado works for r() and e() matrices

    local col `column'

    foreach curdim in row col {
        if "``curdim''"!="" {
            local dimsize = `curdim'sof(`matname')
            if `dimsize' < ``curdim'' {
                disp as error `"Argument of option '`curdim'' out of bounds."'
                exit 125
            }
        }
    }
    
    if "`row'`col'"=="" {
        local numrows = rowsof(`matname')
        forvalues i=1/`numrows' {
            
            mata: _mat2mac_extract("`matname'", `i', 0    )  // returns local `matastring'
            local matastring : subinstr local matastring " " ", " , all

            if `numrows'==1 {
                local matstr "( `matastring' )"
            }
            else if `i'==1 {
                local matstr "( `matastring' \ "
            }
            else if `i'<`numrows' {
                local matstr  "`matstr'`matastring' \ "
            }
            else {
                local matstr  "`matstr'`matastring' )"
            }
        }
    }
    else if "`row'"=="" {
        mata: _mat2mac_extract("`matname'",     0, `col')
        local matstr `matastring'
        if "`comma'"!="" local matstr : subinstr local matstr " " ", " , all
    }
    else if "`col'"=="" {
        mata: _mat2mac_extract("`matname'", `row', 0    )
        local matstr `matastring'
        if "`comma'"!="" local matstr : subinstr local matstr " " ", " , all
    }
    else {
        mata: _mat2mac_extract("`matname'", `row', `col')
        local matstr `matastring'
    }
    
    return local mat2mac `matstr'
    
    if length(`"`matstr'"')>60 local matstr = substr(`"`matstr'"',1,56) + " ..."
    disp as text _n "Extracted: " as result "`matstr'"

end

*** --------------------------------- MATA ------------------------------------------------

version 10.1
mata
mata set matastrict on
void _mat2mac_extract(string scalar matname, real scalar row, real scalar col) {
    // extracts row or column or scalar of a matrix and stores it in macro `matastring'
    
    real scalar size,
                i

    string scalar result

    transmorphic matrix source,
                        submat
    
    source = st_matrix(matname)
    if (row==0) {
        submat = source[., col]'  // note the prime
    }
    else if (col==0) {
        submat = source[row, .]
    }
    else {
        submat = source[row, col]
    }
    
    st_local("matastring", invtokens(strofreal(submat)))
}

end




program define pvalmat, rclass

version 11.0
syntax ,     ///
         Coefmat(namelist min=1 max=1)  /// 
         Varmat(namelist min=1 max=1)  ///
       [ STar                           ///
         Extended                        ///
         se                             ///
         pval]

    if "`se'"!="" & "`pval'"!="" {
        disp as error `"Options 'se' and 'pval' are mutually exclusive."'
        exit 198
    }
    if "`extended'"!="" & "`star'"=="" {
        disp as error `"Option 'extended' is only allowed in conjunction with option 'star'."'
        exit 198
    }
    
    capture confirm matrix `coefmat'
    capture confirm matrix `varmat'

    local rdim1 = `=rowsof(`coefmat')'
    local rdim2 = `=rowsof(`varmat')'
    local cdim1 = `=colsof(`coefmat')'
    local cdim2 = `=colsof(`varmat')'
    
    if (`rdim1'!=`rdim2' | `cdim1'!=`cdim2') {
        disp as error `"Matrices `coefmat' and `varmat' must have the same dimensions."'
        exit 503
    }

    local alt ""
    if "`se'"  !="" local alt se
    if "`pval'"!="" local alt pval

    if "`extended'"!="" local star starext

    local calcpmat_err ""
    tempname mergemat pvalmat
    mata: calcpmat("`mergemat'","`pvalmat'","`coefmat'","`varmat'","`star'","`alt'")
    if "`calcpmat_err'"!="" {
        disp as error `"`calcpmat_err'"'
        exit 508
    }

    local    cnames: colfullnames `coefmat'
    local    rnames: rowfullnames `coefmat'

    local newrnames ""
    foreach currowname of local rnames {
        local newrnames "`newrnames' `currowname' p "
    }

    matrix rownames `mergemat' = `newrnames'
    matrix colnames `mergemat' = `cnames'
    matrix rownames `pvalmat'  = `rnames'
    matrix colnames `pvalmat'  = `cnames'


    return matrix pvalmat  `pvalmat'
    return matrix mergemat `mergemat'
    
end


*** --------------------------------- MATA ------------------------------------------------
version 11.0
mata:
mata set matastrict on
void calcpmat(string scalar mergematname, string scalar pvalmatname, string scalar coefmatname, string scalar varmatname, string scalar star, string scalar alt) {

// also performs checks on input matrices:
// - variance and standard error matrices must have non-negative elems
// - p-value matrix elems must be [0,1]
// if checks fail, returns local calcpmat_err

    real matrix mergemat,
                pvalmat,
                coefmat,
                varmat

    real scalar i,
                j,
                dfr

    coefmat   = st_matrix(coefmatname)
    varmat    = st_matrix(varmatname)
    
    // checks
    if (alt=="") {
        if (sum(varmat:<0)>0) {
            st_local("calcpmat_err","Input matrix must have non-negative elements")
            return
        }
    }
    else if (alt=="se") {
        if (sum(varmat:<0)>0) {
            st_local("calcpmat_err","Input matrix must have non-negative elements")
            return
        }
    }
    else if (alt=="pval") {
        if ( (sum(varmat:<0) + sum( (varmat:>1) :& (varmat:<.) ) ) >0 )  {
            st_local("calcpmat_err","Input matrix must have elements in the unit interval")
            return
        }
    }    
    
    // calculations
    dfr = st_numscalar("e(df_r)")
    
    if (alt=="") {          // varmat passed has variances
        
        varmat = sqrt(varmat)
        
        if (dfr==J(0,0,.)) {
            pvalmat = 2 * ( normal(     -1 * abs(coefmat :/ varmat)) )  // precision note: normal(-x) instead of 1-normal(x)
        }
        else {
            pvalmat = 2 * (  ttail(dfr,      abs(coefmat :/ varmat)) )
        }
    }
    else if (alt=="se") {   // varmat passed has standard errors already
        if (dfr==J(0,0,.)) {
            pvalmat = 2 * ( normal(     -1 * abs(coefmat :/ varmat)) )
        }
        else {
            pvalmat = 2 * (  ttail(dfr,      abs(coefmat :/ varmat)) )
        }
    }
    else if (alt=="pval") { // varmat passed has p-values already
        pvalmat = varmat
    }

    if (star!="") {
        for (i=1;i<=rows(coefmat);i++) {
            for (j=1;j<=cols(coefmat);j++) {
                if (pvalmat[i,j]<=0.01) {
                    pvalmat[i,j]=.a
                }
                else if (pvalmat[i,j]<=0.05) {
                    pvalmat[i,j]=.b
                }
                else if (pvalmat[i,j]<=0.1) {
                    pvalmat[i,j]=.c
                }
                else {
                    if (star=="star") {
                        pvalmat[i,j]=.
                    }
                }
                if (star=="starext") {
                    if (pvalmat[i,j]<=0.25) {
                        pvalmat[i,j]=.d
                    }
                    else if (pvalmat[i,j]<=0.5) {
                        pvalmat[i,j]=.e
                    }
                    else if (pvalmat[i,j]<=1) {
                        pvalmat[i,j]=.
                    }
                }
            }
        }
    }
    mergemat = rowshape( (coefmat , pvalmat) , 2*rows(coefmat) )

    st_matrix(mergematname, mergemat)
    st_matrix(pvalmatname , pvalmat)
}

end



version 10
mata:
mata set matastrict on

void ds_pathparts(string scalar origpath) {
// version 1.0.0  01jul2012  dcs
// break full path into parts: path, filename, root (of filename), extension
// store results in r() macros r(root), r(ext), r(filename), r(path)
// 
// rules
// - to get a r(filename), r(ext), r(root), there must be a dot present in the last element of the string supplied
//   multiple dots in filename are allowed; the last one defines the extension
//   if no dot is present in the last element of the string supplied, everything is packed into r(path)
// - to get r(path), there must either be 
//     no dot in the last elem of the path or
//     if a dot is present, there must be a dir separator
// - if a colon is present, it must be preceeded by some string, otherwise the function errors out
// - the first ending directory separator is removed from r(path); so normally r(path) does not end in a dir separator
//   r(path) only contains a separator for the root dir (e.g. "c:\")
//   it also contains separators if multiple separators are passed
//     i.e. passing "mydir//a.lst" will be split into "mydir/" and "a.lst"
// - r(ext) contains a dot as the first character
// - r-values of missing path parts are not returned (e.g. if only the filename is supplied, r(path) is missing)
// - path may contain blanks
// - dots in paths are allowed

    string scalar path,
                  filename,
                  ext,
                  jnk
    real scalar numdots
    
    pragma unset path
    pragma unset filename
    
    pathsplit(origpath, path, filename)
    ext = pathsuffix(origpath)
    if (ext == "") {     // no extension exists => last elem of path is assumed to be part of directory path and not a file name
        path = pathjoin(path, filename)
        filename = ""
    }
    
    st_rclear()
    st_global("r(path)", path)
    st_global("r(filename)", filename)
    st_global("r(ext)", ext)

    // getting root of filename: account for possibility of several dots in filename
    if (filename != "") {
        jnk = subinstr(filename, ".", "")
        numdots = strlen(filename) - strlen(jnk)
        if (numdots == 0) {
            st_global("r(root)", filename)
        }
        if (numdots == 1 & strpos(filename, ".") > 1) {
            st_global("r(root)", substr(filename, 1, strpos(filename, ".") - 1))
        }
        if (numdots > 1) {
            st_global("r(root)", substr(filename, 1, strlen(filename) - strpos(strreverse(filename), ".")))
        }
    }
}
end


